// Load environment variables FIRST, before any other imports
// Try .env.local first (for development), then fall back to .env
require("dotenv").config({ path: process.env.DOTENV_CONFIG_PATH || ".env.local" });
// Fallback to .env if .env.local doesn't exist or if explicitly set
if (!process.env.DB_HOST && !process.env.DB_USER) {
  require("dotenv").config({ path: ".env" });
}

const express = require("express");
const cors = require("cors");
const frontendRouter = require("./routes/frontend");
const adminRouter = require("./routes/admin");
const mobileRouter = require("./routes/mobile");
const AppError = require("./utils/appError");
const errorHandler = require("./utils/errorHandler");
const session = require("express-session");
const bodyParser = require("body-parser");

const app = express();

// Trust proxy setting for production (behind reverse proxy/load balancer)
// This is required when using express-rate-limit behind proxies
if (app.get("env") === "production") {
  app.set("trust proxy", 1); // Trust first proxy
}

const sessionConfig = {
  secret: "context-app-secret-tokenize-token",
  resave: false,
  saveUninitialized: false,
  // cookie: {
  //     maxAge: 60000,
  //     // sameSite: true,
  //     // httpOnly: true,
  // },
};

const corsOptions = {
  origin: ["http://localhost:3000", "http://localhost:3001"],
  credentials: true,
  optionSuccessStatus: 200,
  // origin: "*",
};

app.use(cors(corsOptions));

if (app.get("env") === "production") {
  sessionConfig.cookie.secure = false; // serve secure cookies
}

// Stripe webhook route needs raw body for signature verification
// This must be before bodyParser.json() middleware
app.use('/api/v1/stripe-webhook', express.raw({ type: 'application/json' }));

app.use(bodyParser.json({ limit: "20mb" }));
app.use((req, res, next) => {
  console.log("Request Method", req?.method);
  console.log("Request URL", req?.url);
  console.log("Request Headers", req?.headers);
  console.log("Request Body", req?.body);
  next();
});

app.use(session(sessionConfig));

// parse requests of content-type - application/json
app.use(express.json({ limit: "50mb" }));

// parse requests of content-type - application/x-www-form-urlencoded
app.use(express.urlencoded({ limit: "50mb", extended: true }));

app.use("/api/v1", frontendRouter);
app.use("/api/mobile/v1", mobileRouter);
app.use("/api/services", adminRouter);

app.all("*", (req, res, next) => {
  next(new AppError(`The URL ${req.originalUrl} does not exists`, 404));
});

app.use(errorHandler);
const PORT = process.env.PORT || 4400;

app.listen(PORT, () => {
  console.log(`server running on port ${PORT}`);
});

module.exports = app;
