#!/usr/bin/env node
/**
 * Quick script to check environment variables
 * Usage: node check-env.js
 */

require("dotenv").config({ path: ".env.local" });

const requiredVars = [
  'DB_HOST',
  'DB_USER', 
  'DB_PASSWORD',
  'DB_NAME',
  'DB_PORT'
];

console.log('\n📋 Environment Variables Status:\n');
console.log('─'.repeat(50));

requiredVars.forEach(varName => {
  const value = process.env[varName];
  // For DB_PASSWORD, empty string is valid (means no password)
  const isValid = varName === 'DB_PASSWORD' 
    ? (value !== undefined && value !== null)  // Empty string is OK
    : (value && value.trim() !== '');          // Other vars must have value
  const status = isValid ? '✅ SET' : '❌ MISSING';
  const displayValue = varName === 'DB_PASSWORD' 
    ? (value !== undefined && value !== null ? '***hidden***' : '(not set)')
    : (value || '(not set)');
  
  console.log(`${varName.padEnd(20)} ${status.padEnd(10)} ${displayValue}`);
});

console.log('─'.repeat(50));

const missing = requiredVars.filter(v => {
  const value = process.env[v];
  if (v === 'DB_PASSWORD') {
    // Empty string is valid for password (means no password)
    return value === undefined || value === null;
  }
  return !value || value.trim() === '';
});
if (missing.length > 0) {
  console.log(`\n⚠️  Missing variables: ${missing.join(', ')}`);
  console.log('\n💡 Add them to your .env.local file!\n');
  process.exit(1);
} else {
  console.log('\n✅ All required variables are set!\n');
  process.exit(0);
}

