const AppError = require("../../utils/appError");
const conn = require("../../services/db").default;
const {
  PAGINATION_LIMIT,
  STATUS_TYPE,
  BOOLEAN,
  MEDIA,
  RICH_TEXT,
} = require("../../config/AppConstant");
const DbHelper = require("../../helpers/DbHelper");
const { base64ToFile } = require("../../lib/FileManager");
const { getSlug } = require("../../helpers/Slug");

const getConfigValue = async (type, config, defaultValue = null) => {
  if (type === MEDIA.toLowerCase()) {
    if (!config) return defaultValue;
    const response = await base64ToFile(config, "config");
    return response?.path || "";
  }
  if (type === BOOLEAN.toLowerCase()) {
    return config === "on" ? "on" : "off";
  }
  return config;
};

module.exports = {
  list: async (req, res, next) => {
    const { title, status, limit, page } = req?.body;
    let query =
      "SELECT SQL_CALC_FOUND_ROWS name,id,status,type, value, created FROM `configs`";
    let queryParams = [];
    const titleReq = title && title?.length > 0 ? title : null;
    const statusReq =
      status && status?.length > 0 && STATUS_TYPE.includes(status)
        ? status
        : null;

    if (titleReq && statusReq) {
      query = query + " WHERE `title` like ? AND `status` = ?";
      queryParams = [`%${titleReq}%`, statusReq];
    } else {
      if (titleReq) {
        query = query + " WHERE `title` like ?";
        queryParams = [`%${titleReq}%`];
      }
      if (statusReq) {
        query = query + " WHERE `status` = ?";
        queryParams = [statusReq];
      }
    }
    const recordLimit = limit || PAGINATION_LIMIT;
    const pageNo = page || 0;
    const OFFSET = pageNo * recordLimit;
    query = query + ` ORDER BY id DESC  LIMIT ?, ?`;
    queryParams.push(OFFSET);
    queryParams.push(recordLimit);

    try {
      conn.query(query, queryParams, async function (err, result, fields) {
        if (err) {
          const errorMessage =
            result?.error?.message || "Something went wrong, Please try again";
          return next(new AppError(errorMessage, 500));
        } else if (result) {
          const response = await DbHelper.promisifyQuery(
            "SELECT COUNT(*) as totalRow FROM `configs`",
            conn,
            next
          );
          const totalRows = (response.length > 0 && response[0]?.totalRow) || 0;
          res.status(200).json({
            status: "success",
            message: "Config Listing",
            result: {
              count: totalRows,
              list: result,
            },
          });
        }
      });
    } catch (e) {
      console.log("Exception Error : Config Listing ", e);
      return next(new AppError("Something went wrong, Please try again", 500));
    }
  },
  save: async (req, res, next) => {
    const { name, type, status, config } = req?.body;
    try {
      const configValue = await getConfigValue(type, config);
      const formData = [
        name,
        type === RICH_TEXT.toLowerCase() ? "richText" : type,
        status ? "active" : "deactive",
        configValue,
      ];
      conn.query(
        "INSERT INTO `configs` (`name`,`type`,`status`,`value`) VALUES (?,?,?,?)",
        formData,
        async function (err, result, fields) {
          if (err) {
            const errorMessage =
              result?.error?.message ||
              "Something went wrong, Please try again";
            return next(new AppError(errorMessage, 500));
          }

          res.status(200).json({
            status: "success",
            message: "Config stored successfully.",
            result,
          });
        }
      );
    } catch (e) {
      console.log("Exception Error : Config Creation ", e);
      return next(new AppError("Something went wrong, Please try again", 500));
    }
  },
  update: async (req, res, next) => {
    try {
      if (!req.params) return next(new AppError("No req data found", 404));
      const { id } = req.params;
      const { name, type, status, config, value } = req?.body;
      const configValue = await getConfigValue(type, config, value);
      let query =
        "UPDATE `configs` SET `name` = ? ,`type` = ?, `value` = ?, `status` = ? WHERE `id` = ?";
      let sqlData = [
        name,
        type,
        configValue,
        status ? "active" : "deactive",
        id,
      ];
      conn.query(query, sqlData, async function (err, result, fields) {
        if (err) {
          const errorMessage =
            result?.error?.message || "Something went wrong, Please try again";
          return next(new AppError(errorMessage, 500));
        }

        res.status(200).json({
          status: "success",
          message: "Config successfully updated.",
          result,
        });
      });
    } catch (e) {
      console.log("Exception Error : Config Updated ", e);
      return next(new AppError("Something went wrong, Please try again", 500));
    }
  },
  view: (req, res, next) => {
    try {
      const { id } = req?.params;
      const query = "SELECT * FROM `configs` WHERE id = ?";
      const sqlData = [id];
      conn.query(query, sqlData, async function (err, result, fields) {
        if (err) {
          const errorMessage =
            result?.error?.message || "Something went wrong, Please try again";
          return next(new AppError(errorMessage, 500));
        }

        res.status(200).json({
          status: "success",
          message: "Config Information fetched",
          result: (result?.length > 0 && result[0]) || undefined,
        });
      });
    } catch (e) {
      console.log("Exception Error : Config Fetched ", e);
      return next(new AppError("Something went wrong, Please try again", 500));
    }
  },
  delete: (req, res, next) => {
    try {
      const { id } = req?.body;
      const query = "DELETE FROM `configs` WHERE `id` = ?";
      const sqlData = [id];
      conn.query(query, sqlData, async function (err, result, fields) {
        if (err) {
          const errorMessage =
            result?.error?.message || "Something went wrong, Please try again";
          return next(new AppError(errorMessage, 500));
        }

        res.status(200).json({
          status: "success",
          message: "Config removed successfully.",
          result: (result?.length > 0 && result[0]) || undefined,
        });
      });
    } catch (e) {
      console.log("Exception Error : Config Removal ", e);
      return next(new AppError("Something went wrong, Please try again", 500));
    }
  },
};
