const DbHelper = require("../../../helpers/DbHelper");
const AppError = require("../../../utils/appError");
const conn = require("../../../services/db");
const { PAGINATION_LIMIT } = require("../../../config/AppConstant");

async function userList(req, res, next) {
  const { name, limit, page } = req?.body;
  let query =
    `SELECT SQL_CALC_FOUND_ROWS 
      u.id, 
      u.first_name, 
      u.last_name, 
      u.full_name, 
      u.email, 
      u.phone, 
      u.shop_name, 
      u.location, 
      u.profile_picture, 
      u.provider, 
      u.created_at,
      sp.id as subscription_plan_id,
      sp.name as subscription_plan_name,
      sp.price as subscription_plan_price,
      sp.duration_type as subscription_duration_type,
      sp.max_locations,
      sp.max_staff,
      us.status as subscription_status,
      us.start_date as subscription_start_date,
      us.end_date as subscription_end_date
    FROM \`users\` u
    LEFT JOIN \`user_subscriptions\` us ON us.user_id = u.id AND us.status = 'active' AND us.end_date > NOW()
    LEFT JOIN \`subscription_plans\` sp ON sp.id = us.subscription_plan_id
    WHERE u.\`is_deleted\` IS NULL`;
  let queryParams = [];
  const nameReq = name ?? null;

  if (nameReq) {
    query = query + " AND (u.`full_name` LIKE ? OR u.`first_name` LIKE ? OR u.`last_name` LIKE ? OR u.`email` LIKE ? OR u.`shop_name` LIKE ?)";
    const searchPattern = `%${nameReq}%`;
    queryParams = [searchPattern, searchPattern, searchPattern, searchPattern, searchPattern];
  }

  const recordLimit = Number(limit) ?? PAGINATION_LIMIT;
  const pageNo = page - 1;

  const OFFSET = pageNo * recordLimit;
  query = query + ` ORDER BY u.id DESC  LIMIT ?, ?`;
  queryParams.push(OFFSET);
  queryParams.push(recordLimit);

  try {
    // Get total count first
    let countQuery = "SELECT COUNT(*) as totalRow FROM `users` u WHERE u.`is_deleted` IS NULL";
    let countParams = [];
    
    if (nameReq) {
      countQuery = countQuery + " AND (u.`full_name` LIKE ? OR u.`first_name` LIKE ? OR u.`last_name` LIKE ? OR u.`email` LIKE ? OR u.`shop_name` LIKE ?)";
      const searchPattern = `%${nameReq}%`;
      countParams = [searchPattern, searchPattern, searchPattern, searchPattern, searchPattern];
    }
    
    const totalRowsResponse = await DbHelper.promisifyQuery(countQuery, conn, next, countParams.length > 0 ? countParams : null);
    const totalRows = (totalRowsResponse.length > 0 && totalRowsResponse[0]?.totalRow) || 0;
    
    // Get user list with subscription data
    conn.query(query, queryParams, async function (err, result, fields) {
      if (err) {
        console.log("Error in Mysql to get list of users", err);
        const errorMessage =
          err?.message || "Something went wrong, Please try again";
        return next(new AppError(errorMessage, 500));
      } else if (result) {
        // Convert subscription_plan_price from string to number (MySQL DECIMAL returns as string)
        const users = (result || []).map(user => ({
          ...user,
          subscription_plan_price: user.subscription_plan_price ? parseFloat(user.subscription_plan_price) || null : null,
        }));
        
        res.status(200).json({
          status: "success",
          message: "Users List",
          total_records: totalRows,
          data: users,
        });
      }
    });
  } catch (e) {
    console.log("Exception Error : Users Listing ", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
}

module.exports = userList;
