const AppError = require("../../../utils/appError");
const conn = require("../../../services/db");

async function updateUser(req, res, next) {
  try {
    const { id } = req.params;
    const {
      firstName,
      lastName,
      email,
      phone,
      shopName,
      location,
    } = req?.body;

    if (!id) {
      return next(new AppError("User ID is required", 400));
    }

    // Trim and validate fields
    const trimmedFirstName = firstName?.trim();
    const trimmedLastName = lastName?.trim();
    const trimmedEmail = email?.trim()?.toLowerCase();
    const trimmedPhone = phone?.trim();
    const trimmedShopName = shopName?.trim();
    const trimmedLocation = location?.trim();

    // Validate required fields
    if (!trimmedFirstName) {
      return next(new AppError("First name is required", 400));
    }
    if (!trimmedLastName) {
      return next(new AppError("Last name is required", 400));
    }
    if (!trimmedEmail) {
      return next(new AppError("Email is required", 400));
    }
    // Validate email format
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(trimmedEmail)) {
      return next(new AppError("Please enter a valid email address", 400));
    }
    if (!trimmedPhone) {
      return next(new AppError("Phone is required", 400));
    }
    if (!trimmedShopName) {
      return next(new AppError("Business is required", 400));
    }
    if (!trimmedLocation) {
      return next(new AppError("Location is required", 400));
    }

    // Construct full_name
    const full_name = `${trimmedFirstName} ${trimmedLastName}`.trim();

    // Check if email already exists for another user
    const checkEmailQuery = "SELECT id FROM `users` WHERE `is_deleted` IS NULL AND email = ? AND id != ?";
    conn.query(
      checkEmailQuery,
      [trimmedEmail, id],
      async function (err, result, fields) {
        if (err) {
          return next(new AppError("Something went wrong, Please try again", 500));
        }
        
        if (result && result.length > 0) {
          return next(new AppError("Email already exists. Please use a different email.", 400));
        }

        // Update user
        const updateQuery =
          "UPDATE `users` SET first_name = ?, last_name = ?, full_name = ?, email = ?, phone = ?, shop_name = ?, location = ? WHERE id = ? AND is_deleted IS NULL";
        
        conn.query(
          updateQuery,
          [trimmedFirstName, trimmedLastName, full_name, trimmedEmail, trimmedPhone, trimmedShopName, trimmedLocation, id],
          async function (err, result, fields) {
            if (err) {
              // Check if error is due to duplicate email (in case of race condition)
              const isDuplicateEntry = 
                err.code === 'ER_DUP_ENTRY' || 
                err.code === 1062 ||
                err.errno === 1062 ||
                (err.message && (
                  err.message.includes('Duplicate entry') || 
                  err.message.includes('Duplicate') ||
                  err.message.toLowerCase().includes('email')
                ));
              
              if (isDuplicateEntry) {
                return next(new AppError("Email already exists. Please use a different email.", 400));
              }
              const errorMessage =
                err?.message || "Something went wrong, Please try again";
              return next(new AppError(errorMessage, 500));
            } else if (result) {
              return res.status(200).json({
                status: "success",
                message: "User updated successfully.",
              });
            }
          }
        );
      }
    );
  } catch (e) {
    console.log("Exception Error : User Update ", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
}

module.exports = updateUser;
