const { query } = require("./shared");
const { recalculateAttendanceHours } = require("./breakHelpers");

async function addBreak(req, res) {
  try {
    const { attendance_id, staff_id, start_time, end_time } = req.body;

    if (!attendance_id || !staff_id || !start_time) {
      return res.status(400).json({
        error: "Attendance ID, staff ID, and start time are required",
      });
    }

    // Verify attendance record exists and belongs to admin
    const attendance = await query(
      `SELECT a.id, a.staff_id, a.attendance_date
      FROM attendance a
      JOIN staff s ON a.staff_id = s.id
      JOIN branches b ON s.branch_id = b.id
      JOIN shops sh ON b.shop_id = sh.id
      WHERE a.id = ? AND sh.admin_id = ?`,
      [attendance_id, req.user.id]
    );

    if (!attendance || attendance.length === 0) {
      return res
        .status(403)
        .json({ error: "Access denied or attendance record not found" });
    }

    const startDateTime = new Date(start_time);
    let endDateTime = end_time ? new Date(end_time) : null;
    let durationMinutes = null;

    if (endDateTime) {
      if (endDateTime < startDateTime) {
        return res
          .status(400)
          .json({ error: "End time must be after start time" });
      }
      durationMinutes = Math.round(
        (endDateTime - startDateTime) / (1000 * 60)
      );
    }

    const result = await query(
      `INSERT INTO work_sessions (attendance_id, staff_id, start_time, end_time, duration_minutes, session_type)
      VALUES (?, ?, ?, ?, ?, 'break')`,
      [attendance_id, staff_id, startDateTime, endDateTime, durationMinutes]
    );

    // Recalculate total hours for the attendance record
    await recalculateAttendanceHours(attendance_id);

    res.status(201).json({
      message: "Break added successfully",
      break_session: {
        id: result.insertId,
        attendance_id,
        start_time: startDateTime.toISOString(),
        end_time: endDateTime?.toISOString() || null,
        duration_minutes: durationMinutes,
      },
    });
  } catch (error) {
    console.error("Add break error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
}

module.exports = addBreak;

