const { query } = require("./shared");

async function breakClockIn(req, res) {
  try {
    let { staff_id: staffId, location } = req.body;
    const requesterRole = req.user.role;

    if (requesterRole === "staff") {
      staffId = req.user.staff_id;
    }

    console.log("\n========================================");
    console.log("☕ BREAK CLOCK IN REQUEST");
    console.log(`👤 Staff ID: ${staffId}`);
    console.log(`📍 Location: ${location || "Not provided"}`);
    console.log(`🕐 Server Time: ${new Date().toISOString()}`);
    console.log("========================================\n");

    if (!staffId) {
      return res.status(400).json({ error: "Staff ID is required" });
    }

    const today = new Date().toISOString().split("T")[0];

    // Find today's attendance record
    const attendance = await query(
      `SELECT id, clock_in_time, clock_out_time
      FROM attendance 
      WHERE staff_id = ? AND attendance_date = ?
      ORDER BY id DESC
      LIMIT 1`,
      [staffId, today]
    );

    if (!attendance || attendance.length === 0) {
      console.log("❌ No clock-in record found for today\n");
      return res
        .status(400)
        .json({ error: "Please clock in first before taking a break" });
    }
    console.log("attendance",attendance);
    const record = attendance[0];

    // Check if already clocked out - handle NULL properly
    const existingClockOutTime = record.clock_out_time;
    const isClockOutNull =
      existingClockOutTime === null ||
      existingClockOutTime === undefined ||
      existingClockOutTime === "null" ||
      existingClockOutTime === "" ||
      (typeof existingClockOutTime === "string" &&
        (existingClockOutTime.trim() === "" ||
          existingClockOutTime.toLowerCase() === "null" ||
          existingClockOutTime.startsWith("0000-00-00")));

    if (!isClockOutNull) {
      console.log("❌ Already clocked out for today\n");
      return res
        .status(400)
        .json({ error: "Cannot take break after clocking out" });
    }

    // Check if there's an active break (break clocked in but not clocked out)
    const activeBreaks = await query(
      `SELECT id FROM work_sessions
      WHERE attendance_id = ? AND session_type = 'break' AND end_time IS NULL
      ORDER BY start_time DESC
      LIMIT 1`,
      [record.id]
    );

    if (activeBreaks && activeBreaks.length > 0) {
      console.log("❌ Already on break\n");
      return res
        .status(400)
        .json({
          error: "Already on break. Please clock out from break first",
        });
    }

    // Get local time string from client or use current time as string
    let breakStartTimeStr = req.body.break_start_time;
    if (!breakStartTimeStr) {
      // If client didn't send time, use current server time as string
      const now = new Date();
      const year = now.getFullYear();
      const month = String(now.getMonth() + 1).padStart(2, "0");
      const day = String(now.getDate()).padStart(2, "0");
      const hours = String(now.getHours()).padStart(2, "0");
      const minutes = String(now.getMinutes()).padStart(2, "0");
      const seconds = String(now.getSeconds()).padStart(2, "0");
      breakStartTimeStr = `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
    }

    // Create break session - store time as string
    const result = await query(
      `INSERT INTO work_sessions 
      (attendance_id, staff_id, start_time, session_type) 
      VALUES (?, ?, ?, 'break')`,
      [record.id, staffId, breakStartTimeStr]
    );

    console.log(`✅ Break clock in successful!`);
    console.log(`   - Break Session ID: ${result.insertId}`);
    console.log(`   - Attendance ID: ${record.id}`);
    console.log(
      `   - Break Start Time (stored as string): ${breakStartTimeStr}`
    );
    console.log("========================================\n");

    res.status(201).json({
      message: "Break started successfully",
      break_session: {
        id: result.insertId,
        attendance_id: record.id,
        start_time: breakStartTimeStr,
      },
    });
  } catch (error) {
    console.error("❌ Break clock in error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
}

module.exports = breakClockIn;

