const { query } = require("./shared");

async function breakClockOut(req, res) {
  try {
    let { staff_id: staffId, location } = req.body;
    const requesterRole = req.user.role;

    if (requesterRole === "staff") {
      staffId = req.user.staff_id;
    }

    console.log("\n========================================");
    console.log("☕ BREAK CLOCK OUT REQUEST");
    console.log(`👤 Staff ID: ${staffId}`);
    console.log(`📍 Location: ${location || "Not provided"}`);
    console.log(`🕐 Server Time: ${new Date().toISOString()}`);
    console.log("========================================\n");

    if (!staffId) {
      return res.status(400).json({ error: "Staff ID is required" });
    }

    const today = new Date().toISOString().split("T")[0];

    // Find today's attendance record - must have clock_in_time and no clock_out_time
    const attendance = await query(
      `SELECT id, clock_in_time, clock_out_time FROM attendance 
      WHERE staff_id = ? AND attendance_date = ? AND clock_in_time IS NOT NULL
      ORDER BY id DESC
      LIMIT 1`,
      [staffId, today]
    );

    if (!attendance || attendance.length === 0) {
      console.log("❌ No active attendance record found for today\n");
      return res.status(400).json({
        error:
          "No active attendance record found for today. Please clock in first.",
      });
    }

    const record = attendance[0];

    // Verify that clock_out_time is NULL (still clocked in)
    const existingClockOutTime = record.clock_out_time;
    const isClockOutNull =
      existingClockOutTime === null ||
      existingClockOutTime === undefined ||
      existingClockOutTime === "null" ||
      existingClockOutTime === "" ||
      (typeof existingClockOutTime === "string" &&
        (existingClockOutTime.trim() === "" ||
          existingClockOutTime.toLowerCase() === "null" ||
          existingClockOutTime.startsWith("0000-00-00")));

    if (!isClockOutNull) {
      console.log("❌ Already clocked out for today - cannot end break\n");
      return res
        .status(400)
        .json({ error: "Cannot end break after clocking out" });
    }

    // Find active break session
    const activeBreaks = await query(
      `SELECT id, start_time FROM work_sessions
      WHERE attendance_id = ? AND session_type = 'break' AND end_time IS NULL
      ORDER BY start_time DESC
      LIMIT 1`,
      [record.id]
    );

    if (!activeBreaks || activeBreaks.length === 0) {
      console.log("❌ No active break found\n");
      return res
        .status(400)
        .json({
          error: "No active break found. Please clock in for break first",
        });
    }

    const breakSession = activeBreaks[0];

    // Get local time string from client or use current time as string
    let breakEndTimeStr = req.body.break_end_time;
    if (!breakEndTimeStr) {
      // If client didn't send time, use current server time as string
      const now = new Date();
      const year = now.getFullYear();
      const month = String(now.getMonth() + 1).padStart(2, "0");
      const day = String(now.getDate()).padStart(2, "0");
      const hours = String(now.getHours()).padStart(2, "0");
      const minutes = String(now.getMinutes()).padStart(2, "0");
      const seconds = String(now.getSeconds()).padStart(2, "0");
      breakEndTimeStr = `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
    }

    // Calculate duration - parse dates more robustly
    let breakStartTime;
    let breakEndTime;
    
    // Parse start_time - handle Date objects, strings, or null
    if (breakSession.start_time instanceof Date) {
      breakStartTime = breakSession.start_time;
    } else if (typeof breakSession.start_time === 'string') {
      // Handle MySQL datetime format (YYYY-MM-DD HH:MM:SS)
      const startTimeStr = breakSession.start_time.trim();
      breakStartTime = new Date(startTimeStr.replace(" ", "T"));
    } else {
      console.error("❌ Invalid start_time format:", breakSession.start_time);
      return res.status(400).json({ error: "Invalid break start time format" });
    }
    
    // Parse end_time - handle string format
    if (typeof breakEndTimeStr === 'string') {
      const endTimeStr = breakEndTimeStr.trim();
      breakEndTime = new Date(endTimeStr.replace(" ", "T"));
    } else {
      console.error("❌ Invalid end_time format:", breakEndTimeStr);
      return res.status(400).json({ error: "Invalid break end time format" });
    }
    
    // Validate dates are valid
    if (Number.isNaN(breakStartTime.getTime()) || Number.isNaN(breakEndTime.getTime())) {
      console.error("❌ Invalid date format for break times");
      console.error(`   Start time: ${breakSession.start_time} (parsed: ${breakStartTime})`);
      console.error(`   End time: ${breakEndTimeStr} (parsed: ${breakEndTime})`);
      return res.status(400).json({ error: "Invalid break time format" });
    }
    
    const breakDurationMs = breakEndTime - breakStartTime;
    const breakDurationMinutes = Math.max(0, Math.round(breakDurationMs / (1000 * 60)));
    
    // Ensure we have a valid number (not NaN)
    if (Number.isNaN(breakDurationMinutes) || !Number.isFinite(breakDurationMinutes)) {
      console.error("❌ Invalid duration calculation");
      return res.status(500).json({ error: "Failed to calculate break duration" });
    }

    // Update break session - store time as string
    await query(
      `UPDATE work_sessions 
      SET end_time = ?, duration_minutes = ?
      WHERE id = ?`,
      [breakEndTimeStr, breakDurationMinutes, breakSession.id]
    );

    console.log(`✅ Break clock out successful!`);
    console.log(`   - Break Session ID: ${breakSession.id}`);
    console.log(`   - Break End Time (stored as string): ${breakEndTimeStr}`);
    console.log(`   - Break Duration: ${breakDurationMinutes} minutes`);
    console.log("========================================\n");

    res.json({
      message: "Break ended successfully",
      break_session: {
        id: breakSession.id,
        end_time: breakEndTimeStr,
        duration_minutes: breakDurationMinutes,
      },
    });
  } catch (error) {
    console.error("❌ Break clock out error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
}

module.exports = breakClockOut;

