const { query } = require("./shared");

// Helper function to recalculate attendance hours after break changes
async function recalculateAttendanceHours(attendanceId) {
  try {
    const attendance = await query(
      `SELECT clock_in_time, clock_out_time FROM attendance WHERE id = ?`,
      [attendanceId]
    );

    if (
      !attendance ||
      attendance.length === 0 ||
      !attendance[0].clock_in_time ||
      !attendance[0].clock_out_time
    ) {
      return; // Can't calculate if no clock out
    }

    const clockInTime = new Date(attendance[0].clock_in_time);
    const clockOutTime = new Date(attendance[0].clock_out_time);
    const totalTimeMs = clockOutTime - clockInTime;

    // Get total break time
    const breaks = await query(
      `SELECT COALESCE(SUM(duration_minutes), 0) as total_break_minutes
      FROM work_sessions
      WHERE attendance_id = ? AND session_type = 'break' AND end_time IS NOT NULL`,
      [attendanceId]
    );

    const totalBreakMinutes = breaks[0]?.total_break_minutes || 0;
    const totalBreakMs = totalBreakMinutes * 60 * 1000;
    const totalHours = (totalTimeMs - totalBreakMs) / (1000 * 60 * 60);

    await query(`UPDATE attendance SET total_hours = ? WHERE id = ?`, [
      totalHours.toFixed(2),
      attendanceId,
    ]);
  } catch (error) {
    console.error("Error recalculating attendance hours:", error);
  }
}

module.exports = { recalculateAttendanceHours };

