const { query } = require("./shared");

async function clockIn(req, res) {
  try {
    let { staff_id: staffId, branch_id: branchId, location, clock_in_time } =
      req.body;
    const requesterRole = req.user.role;

    if (requesterRole === "staff") {
      staffId = req.user.staff_id;
      branchId = req.user.branch_id;
    }

    if (!staffId || !branchId) {
      return res
        .status(400)
        .json({ error: "Staff ID and branch ID are required" });
    }

    // Require explicit clock-in time from client
    if (!clock_in_time) {
      return res
        .status(400)
        .json({ error: "clock_in_time is required" });
    }

    const parsedClockIn = new Date(clock_in_time);
    if (Number.isNaN(parsedClockIn.getTime())) {
      return res
        .status(400)
        .json({ error: "Invalid clock_in_time format" });
    }

    const attendanceDate = parsedClockIn.toISOString().split("T")[0];

    let staffRows;
    if (requesterRole === "staff") {
      staffRows = await query(
        `
        SELECT st.id, st.full_name
        FROM staff st
        WHERE st.id = ? AND st.branch_id = ?
      `,
        [staffId, branchId]
      );
    } else {
      staffRows = await query(
        `
        SELECT st.id, st.full_name
        FROM staff st
        JOIN branches b ON st.branch_id = b.id
        JOIN shops s ON b.shop_id = s.id
        WHERE st.id = ? AND b.id = ? AND s.admin_id = ?
      `,
        [staffId, branchId, req.user.id]
      );
    }

    if (!staffRows || staffRows.length === 0) {
      console.log(
        "❌ Access denied - staff not found or doesn't belong to admin's branch\n"
      );
      return res
        .status(403)
        .json({ error: "Access denied or staff not found" });
    }

    const staff = staffRows[0];

    const existing = await query(
      `SELECT id, clock_out_time FROM attendance 
       WHERE staff_id = ? AND attendance_date = ?`,
      [staffId, attendanceDate]
    );

    console.log(`📋 Existing attendance records for today: ${existing.length}`);
    if (existing.length > 0) {
      console.log(`   - Record ID: ${existing[0].id}`);
      console.log(
        `   - Clock Out Time: ${
          existing[0].clock_out_time || "NULL (still clocked in)"
        }`
      );
    }

    if (existing.length > 0 && !existing[0].clock_out_time) {
      return res.status(400).json({ error: "Already clocked in for today" });
    }
    const result = await query(
      `INSERT INTO attendance 
       (staff_id, branch_id, clock_in_time, clock_in_location, attendance_date, status) 
       VALUES (?, ?, ?, ?, ?, 'present')`,
      [staffId, branchId, parsedClockIn, location || null, attendanceDate]
    );

    res.status(201).json({
      message: "Clocked in successfully",
      attendance: {
        id: result.insertId,
        staff_id: staffId,
        branch_id: branchId,
        clock_in_time: parsedClockIn,
        attendance_date: attendanceDate,
      },
    });
  } catch (error) {
    console.error("❌ Clock in error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
}

module.exports = clockIn;

