const { query } = require("./shared");

async function clockOut(req, res) {
  try {
    let { staff_id: staffId, location, clock_out_time } = req.body;
    const requesterRole = req.user.role;

    if (requesterRole === "staff") {
      staffId = req.user.staff_id;
    }

    if (!staffId) {
      return res.status(400).json({ error: "Staff ID is required" });
    }

    // Require explicit clock-out time from client
    if (!clock_out_time) {
      return res
        .status(400)
        .json({ error: "clock_out_time is required" });
    }

    // Parse and format clock_out_time the same way as clock_in_time
    const parsedClockOut = new Date(clock_out_time);
    if (Number.isNaN(parsedClockOut.getTime())) {
      return res
        .status(400)
        .json({ error: "Invalid clock_out_time format" });
    }

    // Format clock_out_time for MySQL using LOCAL time components (YYYY-MM-DD HH:MM:SS)
    // Same logic as clockIn.js
    const year = parsedClockOut.getFullYear();
    const month = String(parsedClockOut.getMonth() + 1).padStart(2, '0');
    const day = String(parsedClockOut.getDate()).padStart(2, '0');
    const hours = String(parsedClockOut.getHours()).padStart(2, '0');
    const minutes = String(parsedClockOut.getMinutes()).padStart(2, '0');
    const seconds = String(parsedClockOut.getSeconds()).padStart(2, '0');
    const formattedClockOut = `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;

    let staffRows;
    if (requesterRole === "staff") {
      staffRows = await query(
        `
        SELECT st.id, st.branch_id
        FROM staff st
        WHERE st.id = ?
      `,
        [staffId]
      );
    } else {
      staffRows = await query(
        `
        SELECT st.id
        FROM staff st
        JOIN branches b ON st.branch_id = b.id
        JOIN shops s ON b.shop_id = s.id
        WHERE st.id = ? AND s.admin_id = ?
      `,
        [staffId, req.user.id]
      );
    }

    if (!staffRows || staffRows.length === 0) {
      return res.status(403).json({ error: "Access denied or staff not found" });
    }

    // Avoid zero-date literal to prevent strict mode errors; treat NULL or very-early dates as open
    const attendance = await query(
      `SELECT id, clock_in_time, clock_out_time, attendance_date
       FROM attendance 
       WHERE staff_id = ? 
         AND (clock_out_time IS NULL OR clock_out_time <= '1971-01-01 00:00:00')
       ORDER BY clock_in_time DESC
       LIMIT 1`,
      [staffId]
    );

    if (!attendance || attendance.length === 0) {
      return res.status(400).json({ error: "No open clock-in record found" });
    }

    const record = attendance[0];
    console.log("📋 Found attendance record:", record);

    const clockInTime = new Date(record.clock_in_time);
    // Parse formattedClockOut back to Date for calculation (treat as local time)
    const clockOutTime = new Date(formattedClockOut.replace(' ', 'T'));
    const totalHours = Math.max(
      0,
      (clockOutTime - clockInTime) / (1000 * 60 * 60)
    );
    
    console.log(`🕐 Received clock_out_time: ${clock_out_time}`);
    console.log(`🕐 Formatted for DB: ${formattedClockOut}`);
    console.log(`⏱️  Total Hours: ${totalHours.toFixed(2)}`);

    const updateResult = await query(
      `UPDATE attendance 
       SET clock_out_time = ?, 
           clock_out_location = ?, 
           total_hours = ?
       WHERE id = ?`,
      [formattedClockOut, location || null, totalHours.toFixed(2), record.id]
    );

    console.log(`✅ Update result:`, updateResult);
    console.log(`📝 Affected rows: ${updateResult.affectedRows}`);

    if (updateResult.affectedRows === 0) {
      console.error("❌ No rows were updated!");
      return res.status(500).json({ error: "Failed to update clock-out record" });
    }

    res.json({
      message: "Clocked out successfully",
      attendance: {
        id: record.id,
        clock_out_time: formattedClockOut,
        total_hours: totalHours.toFixed(2),
      },
    });
  } catch (error) {
    console.error("❌ Clock out error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
}

module.exports = clockOut;

