const { query } = require("./shared");

async function createOrUpdateManualAttendance(req, res) {
  try {
    const {
      staff_id,
      branch_id,
      attendance_date,
      clock_in_time,
      clock_out_time,
      location,
    } = req.body;

    if (!staff_id || !branch_id || !attendance_date) {
      return res.status(400).json({
        error: "Staff ID, branch ID, and attendance date are required",
      });
    }

    if (!clock_in_time && !clock_out_time) {
      return res.status(400).json({
        error: "At least clock-in or clock-out time must be provided",
      });
    }

    const staffRows = await query(
      `
      SELECT st.id, st.full_name
      FROM staff st
      JOIN branches b ON st.branch_id = b.id
      JOIN shops s ON b.shop_id = s.id
      WHERE st.id = ? AND b.id = ? AND s.admin_id = ?
    `,
      [staff_id, branch_id, req.user.id]
    );

    if (!staffRows || staffRows.length === 0) {
      return res.status(403).json({ error: "Access denied or staff not found" });
    }
    const staff = staffRows[0];

    const existingRecords = await query(
      `SELECT id, clock_in_time, clock_out_time, total_hours 
       FROM attendance 
       WHERE staff_id = ? AND attendance_date = ?`,
      [staff_id, attendance_date]
    );

    let clockInDateTime = null;
    let clockOutDateTime = null;
    let totalHours = null;

    if (clock_in_time) {
      clockInDateTime = new Date(clock_in_time);
      if (Number.isNaN(clockInDateTime.getTime())) {
        return res.status(400).json({ error: "Invalid clock-in time format" });
      }
    }

    if (clock_out_time) {
      clockOutDateTime = new Date(clock_out_time);
      if (Number.isNaN(clockOutDateTime.getTime())) {
        return res.status(400).json({ error: "Invalid clock-out time format" });
      }
    }

    if (clockInDateTime && clockOutDateTime) {
      if (clockOutDateTime < clockInDateTime) {
        return res
          .status(400)
          .json({ error: "Clock-out time must be after clock-in time" });
      }
      totalHours = (clockOutDateTime - clockInDateTime) / (1000 * 60 * 60);
    } else if (
      existingRecords.length > 0 &&
      existingRecords[0].clock_in_time &&
      clockOutDateTime
    ) {
      const existingClockIn = new Date(existingRecords[0].clock_in_time);
      if (clockOutDateTime < existingClockIn) {
        return res
          .status(400)
          .json({ error: "Clock-out time must be after clock-in time" });
      }
      totalHours = (clockOutDateTime - existingClockIn) / (1000 * 60 * 60);
    } else if (
      clockInDateTime &&
      existingRecords.length > 0 &&
      existingRecords[0].clock_out_time
    ) {
      const existingClockOut = new Date(existingRecords[0].clock_out_time);
      if (existingClockOut < clockInDateTime) {
        return res
          .status(400)
          .json({ error: "Clock-in time must be before clock-out time" });
      }
      totalHours = (existingClockOut - clockInDateTime) / (1000 * 60 * 60);
    }

    if (existingRecords.length > 0) {
      const updateFields = [];
      const updateValues = [];

      if (clock_in_time !== undefined) {
        updateFields.push("clock_in_time = ?");
        updateValues.push(clockInDateTime);
        if (location) {
          updateFields.push("clock_in_location = ?");
          updateValues.push(location);
        }
      }

      if (clock_out_time !== undefined) {
        updateFields.push("clock_out_time = ?");
        updateValues.push(clockOutDateTime);
        if (location) {
          updateFields.push("clock_out_location = ?");
          updateValues.push(location);
        }
      }

      if (totalHours !== null) {
        updateFields.push("total_hours = ?");
        updateValues.push(totalHours.toFixed(2));
      }

      if (clockInDateTime) {
        updateFields.push("status = ?");
        updateValues.push("present");
      }

      updateValues.push(existingRecords[0].id);

      const updateQuery = `UPDATE attendance SET ${updateFields.join(
        ", "
      )} WHERE id = ?`;
      await query(updateQuery, updateValues);

      const staff = staffRows[0];

      res.json({
        message: "Attendance updated successfully",
        attendance: {
          id: existingRecords[0].id,
          staff_id,
          branch_id,
          attendance_date,
          clock_in_time:
            clockInDateTime || existingRecords[0].clock_in_time,
          clock_out_time:
            clockOutDateTime || existingRecords[0].clock_out_time,
          total_hours:
            totalHours !== null
              ? totalHours.toFixed(2)
              : existingRecords[0].total_hours,
        },
      });
    } else {
      const result = await query(
        `INSERT INTO attendance 
         (staff_id, branch_id, attendance_date, clock_in_time, clock_out_time, 
          clock_in_location, clock_out_location, total_hours, status) 
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [
          staff_id,
          branch_id,
          attendance_date,
          clockInDateTime,
          clockOutDateTime,
          location || null,
          location || null,
          totalHours !== null ? totalHours.toFixed(2) : null,
          clockInDateTime ? "present" : "absent",
        ]
      );

      res.status(201).json({
        message: "Attendance created successfully",
        attendance: {
          id: result.insertId,
          staff_id,
          branch_id,
          attendance_date,
          clock_in_time: clockInDateTime,
          clock_out_time: clockOutDateTime,
          total_hours: totalHours !== null ? totalHours.toFixed(2) : null,
        },
      });
    }
  } catch (error) {
    res.status(500).json({ error: "Internal server error" });
  }
}

module.exports = createOrUpdateManualAttendance;

