const { query } = require("./shared");
const encryptionService = require("../../../services/encryptionService");

const normalizeDateTime = (value) => {
  if (!value) return null;

  if (value instanceof Date) {
    if (Number.isNaN(value.getTime())) {
      return null;
    }
    // CRITICAL FIX: MySQL returns Date objects, but we need to extract LOCAL time components
    // to preserve the exact time stored in the database without timezone conversion
    // Example: If DB has "2025-12-08 15:25:00", we want to return "2025-12-08 15:25:00" not "2025-12-08T04:11:54.000Z"
    const year = value.getFullYear();
    const month = String(value.getMonth() + 1).padStart(2, '0');
    const day = String(value.getDate()).padStart(2, '0');
    const hours = String(value.getHours()).padStart(2, '0');
    const minutes = String(value.getMinutes()).padStart(2, '0');
    const seconds = String(value.getSeconds()).padStart(2, '0');
    // Return as local time string format: "YYYY-MM-DD HH:MM:SS"
    return `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
  }

  if (typeof value === 'string') {
    const trimmed = value.trim();
    if (!trimmed || trimmed.toLowerCase() === 'null' || trimmed === 'undefined') {
      return null;
    }

    const zeroDatePatterns = [
      '0000-00-00 00:00:00',
      '0000-00-00T00:00:00',
      '0000-00-00T00:00:00.000Z',
      '1970-01-01 00:00:00',
      '1970-01-01T00:00:00',
    ];

    if (zeroDatePatterns.includes(trimmed)) {
      return null;
    }

    // CRITICAL FIX: Database stores times as local time strings (e.g., "2025-12-05 13:26:09")
    // We should return them as-is WITHOUT converting to UTC
    // This ensures the app receives the exact server time from the database
    // If the string is already in ISO format with timezone (ends with Z or has +/-), extract local components
    if (trimmed.endsWith('Z') || trimmed.includes('+') || (trimmed.includes('-') && trimmed.length > 19)) {
      // Parse ISO format and extract local time components
      try {
        const date = new Date(trimmed);
        if (!Number.isNaN(date.getTime())) {
          const year = date.getFullYear();
          const month = String(date.getMonth() + 1).padStart(2, '0');
          const day = String(date.getDate()).padStart(2, '0');
          const hours = String(date.getHours()).padStart(2, '0');
          const minutes = String(date.getMinutes()).padStart(2, '0');
          const seconds = String(date.getSeconds()).padStart(2, '0');
          return `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
        }
      } catch (e) {
        // Fall through to return as-is
      }
      return trimmed;
    }

    // Local time string format (e.g., "2025-12-05 13:26:09" or "2025-12-05T13:26:09")
    // Normalize T to space for consistency
    return trimmed.replace('T', ' ');
  }

  if (typeof value === 'number') {
    if (value <= 0) {
      return null;
    }
    // For timestamps, extract local time components (not UTC)
    const dateFromMillis = new Date(value);
    if (Number.isNaN(dateFromMillis.getTime())) {
      return null;
    }
    const year = dateFromMillis.getFullYear();
    const month = String(dateFromMillis.getMonth() + 1).padStart(2, '0');
    const day = String(dateFromMillis.getDate()).padStart(2, '0');
    const hours = String(dateFromMillis.getHours()).padStart(2, '0');
    const minutes = String(dateFromMillis.getMinutes()).padStart(2, '0');
    const seconds = String(dateFromMillis.getSeconds()).padStart(2, '0');
    return `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
  }

  return null;
};

const normalizeDate = (value) => {
  if (!value) return null;

  // IMPORTANT:
  // Never use toISOString() for DATE fields coming from MySQL, because that
  // converts the value to UTC and can shift the calendar day (e.g. 2025‑11‑20
  // in local time becomes 2025‑11‑19T18:30:00.000Z, which then formats as
  // 2025‑11‑19). We always want the exact stored calendar date.

  if (value instanceof Date) {
    if (Number.isNaN(value.getTime())) {
      return null;
    }
    // Use local date parts so the calendar day matches the DB value
    const year = value.getFullYear();
    const month = String(value.getMonth() + 1).padStart(2, '0');
    const day = String(value.getDate()).padStart(2, '0');
    return `${year}-${month}-${day}`;
  }

  if (typeof value === 'string') {
    const trimmed = value.trim();
    if (!trimmed || trimmed.toLowerCase() === 'null' || trimmed === 'undefined') {
      return null;
    }
    // If a time part exists, just drop it – the date portion is authoritative
    return trimmed.split('T')[0];
  }

  return null;
};

const normalizeTotalHours = (value) => {
  if (value === null || value === undefined) {
    return null;
  }
  if (typeof value === "number") {
    return Number.isFinite(value) ? Number(value.toFixed(2)) : null;
  }
  if (typeof value === "string") {
    const parsed = parseFloat(value);
    return Number.isNaN(parsed) ? null : Number(parsed.toFixed(2));
  }
  return null;
};

async function getBranchReports(req, res) {
  try {
    const { branch_id } = req.params;
    const { start_date, end_date } = req.query;

    console.log('\n========================================');
    console.log(`📊 FETCHING REPORTS FOR BRANCH ${branch_id}`);
    console.log(`📅 Date Range: ${start_date} to ${end_date}`);
    console.log(`🕐 Server Time: ${new Date().toISOString()}`);
    console.log(`📆 Server Date (Local): ${new Date().toISOString().split('T')[0]}`);
    console.log('========================================\n');

    // Verify branch belongs to this admin
    const branches = await query(
      `
      SELECT b.id, b.name, s.name as shop_name
      FROM branches b
      JOIN shops s ON b.shop_id = s.id
      WHERE b.id = ? AND s.admin_id = ?
    `,
      [branch_id, req.user.id]
    );

    if (!branches || branches.length === 0) {
      return res.status(403).json({ error: 'Access denied to this branch' });
    }

    console.log(`✅ Branch verified: ${branches[0].name} (Shop: ${branches[0].shop_name})`);
    console.log(`👤 Admin ID: ${req.user.id}\n`);

    // Build date condition
    let dateCondition = '';
    const params = [branch_id];
    
    if (start_date && end_date) {
      dateCondition = 'AND a.attendance_date BETWEEN ? AND ?';
      params.push(start_date, end_date);
    } else if (start_date) {
      dateCondition = 'AND a.attendance_date >= ?';
      params.push(start_date);
    } else if (end_date) {
      dateCondition = 'AND a.attendance_date <= ?';
      params.push(end_date);
    }

    // Get all staff in the branch
    const staffRaw = await query(
      `
      SELECT id, staff_code, full_name, email, phone, position, hourly_rate, profile_image_path, created_at
      FROM staff
      WHERE branch_id = ? AND is_active = true
      ORDER BY full_name
    `,
      [branch_id]
    );

    // Decrypt staff data
    const staff = staffRaw.map(s => {
      try {
        s.full_name = s.full_name ? encryptionService.decrypt(s.full_name) : null;
        s.email = s.email ? encryptionService.decrypt(s.email) : null;
        s.phone = s.phone ? encryptionService.decrypt(s.phone) : null;
      } catch (error) {
        console.warn(`Failed to decrypt data for staff ${s.id}:`, error.message);
      }
      return s;
    });

    // Get attendance records
    const attendanceParams = [branch_id];
    let attendanceQuery = `
      SELECT 
        a.*,
        s.staff_code,
        s.full_name,
        s.position
      FROM attendance a
      JOIN staff s ON a.staff_id = s.id
      WHERE a.branch_id = ?
    `;
    
    if (start_date && end_date) {
      attendanceQuery += ' AND a.attendance_date BETWEEN ? AND ?';
      attendanceParams.push(start_date, end_date);
    } else if (start_date) {
      attendanceQuery += ' AND a.attendance_date >= ?';
      attendanceParams.push(start_date);
    } else if (end_date) {
      attendanceQuery += ' AND a.attendance_date <= ?';
      attendanceParams.push(end_date);
    }
    
    attendanceQuery += ' ORDER BY a.id DESC, a.clock_in_time DESC';

    console.log('📝 Attendance Query:');
    console.log(attendanceQuery);
    console.log('📝 Query Params:', attendanceParams);
    console.log('');

    const attendanceRaw = await query(attendanceQuery, attendanceParams);

    // Get break sessions for all attendance records
    const attendanceIds = attendanceRaw.map(r => r.id);
    let breaksMap = {};

    if (attendanceIds.length > 0) {
      const placeholders = attendanceIds.map(() => '?').join(',');
      const breaks = await query(
        `SELECT 
          id,
          attendance_id,
          start_time,
          end_time,
          duration_minutes,
          session_type
        FROM work_sessions
        WHERE attendance_id IN (${placeholders}) AND session_type = 'break'
        ORDER BY start_time ASC`,
        attendanceIds
      );
      
      // Group breaks by attendance_id
      breaks.forEach(breakRecord => {
        const attId = breakRecord.attendance_id;
        if (!breaksMap[attId]) {
          breaksMap[attId] = [];
        }
        breaksMap[attId].push({
          id: breakRecord.id,
          start_time: normalizeDateTime(breakRecord.start_time),
          end_time: normalizeDateTime(breakRecord.end_time),
          duration_minutes: breakRecord.duration_minutes || 0,
        });
      });
    }

    const attendance = attendanceRaw.map(record => {
      const normalizedClockIn = normalizeDateTime(record.clock_in_time);
      const normalizedClockOut = normalizeDateTime(record.clock_out_time);
      const normalizedCreatedAt = normalizeDateTime(record.created_at);
      const normalizedUpdatedAt = normalizeDateTime(record.updated_at);

      // Decrypt staff name in attendance record
      let decryptedName = record.full_name;
      try {
        if (record.full_name) {
          decryptedName = encryptionService.decrypt(record.full_name);
        }
      } catch (error) {
        // console.warn(`Failed to decrypt name for attendance ${record.id}:`, error.message);
      }

      // Get breaks for this attendance record
      const breaks = breaksMap[record.id] || [];
      const totalBreakMinutes = breaks.reduce((sum, b) => sum + (b.duration_minutes || 0), 0);

      return {
        ...record,
        full_name: decryptedName, // Use decrypted name
        clock_in_time: normalizedClockIn,
        clock_out_time: normalizedClockOut,
        attendance_date: normalizeDate(record.attendance_date),
        created_at: normalizedCreatedAt,
        updated_at: normalizedUpdatedAt,
        total_hours: normalizeTotalHours(record.total_hours),
        is_working_now: Boolean(normalizedClockIn && !normalizedClockOut),
        is_automatic_clockout: Boolean(record.is_automatic_clockout),
        breaks: breaks,
        total_break_minutes: totalBreakMinutes,
      };
    });

    console.log(`📋 Total attendance records found: ${attendance.length}`);
    
    // Debug: Show today's records specifically
    const today = new Date().toISOString().split('T')[0];
    const todayRecords = attendance.filter(a => a.attendance_date === today);
    console.log(`📅 Records for TODAY (${today}): ${todayRecords.length}`);
    
    // Debug: Show ongoing sessions
    const ongoingSessions = attendance.filter(a => a.is_working_now);
    console.log(`🔴 Ongoing sessions (no clock-out): ${ongoingSessions.length}`);
    
    if (ongoingSessions.length > 0) {
      console.log('\n📌 Details of ongoing sessions:');
      ongoingSessions.forEach((session, idx) => {
        console.log(`   ${idx + 1}. Staff: ${session.full_name} (ID: ${session.staff_id})`);
        console.log(`      Date: ${session.attendance_date}`);
        console.log(`      Clock In: ${session.clock_in_time}`);
        console.log(`      Clock Out: ${session.clock_out_time || 'NULL (still working)'}`);
        console.log(`      Status: ${session.status}`);
        console.log('');
      });
    } else {
      console.log('⚠️  No ongoing sessions found!\n');
    }

    // Calculate statistics per staff
    const staffStats = {};
    staff.forEach(s => {
      staffStats[s.id] = {
        ...s,
        total_hours: 0,
        session_count: 0,
        present_days: 0,
        absent_days: 0,
        late_days: 0,
        hourly_rate: s.hourly_rate || 0
      };
    });

    attendance.forEach(record => {
      if (staffStats[record.staff_id]) {
        staffStats[record.staff_id].session_count++;
        
        // Only count hours if NOT an automatic clockout
        if (record.total_hours && !record.is_automatic_clockout) {
          staffStats[record.staff_id].total_hours += record.total_hours;
        }
        
        if (record.status === 'present') {
          staffStats[record.staff_id].present_days++;
        } else if (record.status === 'absent') {
          staffStats[record.staff_id].absent_days++;
        } else if (record.status === 'late') {
          staffStats[record.staff_id].late_days++;
        }
      }
    });

    // Overall summary
    // Exclude automatic clockouts from total hours calculation
    const summary = {
      branch: branches[0],
      total_staff: staff.length,
      total_attendance_records: attendance.length,
      total_hours_worked: attendance.reduce((sum, a) => {
        // Only count hours if NOT an automatic clockout
        if (a.is_automatic_clockout) return sum;
        return sum + (a.total_hours || 0);
      }, 0),
      average_hours_per_day: 0,
      present_count: attendance.filter(a => a.status === 'present').length,
      absent_count: attendance.filter(a => a.status === 'absent').length,
      late_count: attendance.filter(a => a.status === 'late').length
    };

    if (summary.total_attendance_records > 0) {
      summary.average_hours_per_day = summary.total_hours_worked / summary.total_attendance_records;
    }

    res.json({
      branch: branches[0],
      staff: Object.values(staffStats),
      attendance,
      summary,
      date_range: {
        start: start_date || null,
        end: end_date || null
      }
    });
  } catch (error) {
    console.error('Get branch reports error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
}

module.exports = getBranchReports;

