const { query } = require("./shared");

const normalizeDateTime = (value) => {
  if (!value) return null;

  if (value instanceof Date) {
    if (Number.isNaN(value.getTime())) {
      return null;
    }
    // Extract LOCAL time components to preserve the exact time stored in the database
    const year = value.getFullYear();
    const month = String(value.getMonth() + 1).padStart(2, "0");
    const day = String(value.getDate()).padStart(2, "0");
    const hours = String(value.getHours()).padStart(2, "0");
    const minutes = String(value.getMinutes()).padStart(2, "0");
    const seconds = String(value.getSeconds()).padStart(2, "0");
    return `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
  }

  if (typeof value === "string") {
    const trimmed = value.trim();
    if (
      !trimmed ||
      trimmed.toLowerCase() === "null" ||
      trimmed === "undefined"
    ) {
      return null;
    }

    const zeroDatePatterns = [
      "0000-00-00 00:00:00",
      "0000-00-00T00:00:00",
      "0000-00-00T00:00:00.000Z",
      "1970-01-01 00:00:00",
      "1970-01-01T00:00:00",
    ];

    if (zeroDatePatterns.includes(trimmed)) {
      return null;
    }

    // If the string is already in ISO format with timezone, extract local components
    if (
      trimmed.endsWith("Z") ||
      trimmed.includes("+") ||
      (trimmed.includes("-") && trimmed.length > 19)
    ) {
      try {
        const date = new Date(trimmed);
        if (!Number.isNaN(date.getTime())) {
          const year = date.getFullYear();
          const month = String(date.getMonth() + 1).padStart(2, "0");
          const day = String(date.getDate()).padStart(2, "0");
          const hours = String(date.getHours()).padStart(2, "0");
          const minutes = String(date.getMinutes()).padStart(2, "0");
          const seconds = String(date.getSeconds()).padStart(2, "0");
          return `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
        }
      } catch (e) {
        // Fall through to return as-is
      }
      return trimmed;
    }

    // Local time string format - normalize T to space for consistency
    return trimmed.replace("T", " ");
  }

  if (typeof value === "number") {
    if (value <= 0) {
      return null;
    }
    const dateFromMillis = new Date(value);
    if (Number.isNaN(dateFromMillis.getTime())) {
      return null;
    }
    const year = dateFromMillis.getFullYear();
    const month = String(dateFromMillis.getMonth() + 1).padStart(2, "0");
    const day = String(dateFromMillis.getDate()).padStart(2, "0");
    const hours = String(dateFromMillis.getHours()).padStart(2, "0");
    const minutes = String(dateFromMillis.getMinutes()).padStart(2, "0");
    const seconds = String(dateFromMillis.getSeconds()).padStart(2, "0");
    return `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
  }

  return null;
};

async function getBreaks(req, res) {
  try {
    const { attendance_id } = req.params;

    console.log(`\n🔍 GET BREAKS REQUEST:`);
    console.log(`   Attendance ID: ${attendance_id}`);

    const breaksRaw = await query(
      `SELECT id, start_time, end_time, duration_minutes
      FROM work_sessions
      WHERE attendance_id = ? AND session_type = 'break'
      ORDER BY start_time ASC`,
      [attendance_id]
    );

    console.log(`   Raw breaks found: ${breaksRaw ? breaksRaw.length : 0}`);
    if (breaksRaw && breaksRaw.length > 0) {
      breaksRaw.forEach((b, idx) => {
        console.log(
          `   Break ${idx + 1}: id=${b.id}, start=${b.start_time}, end=${b.end_time}, duration=${b.duration_minutes}`
        );
      });
    }

    const breaks = (breaksRaw || []).map((breakRecord) => ({
      id: breakRecord.id,
      start_time: normalizeDateTime(breakRecord.start_time),
      end_time: normalizeDateTime(breakRecord.end_time),
      duration_minutes: breakRecord.duration_minutes || 0,
    }));

    console.log(`   Normalized breaks: ${breaks.length}`);
    if (breaks.length > 0) {
      breaks.forEach((b, idx) => {
        console.log(
          `   Normalized Break ${idx + 1}: id=${b.id}, start=${b.start_time}, end=${b.end_time}, duration=${b.duration_minutes}`
        );
      });
    }
    console.log("========================================\n");

    res.json({ breaks });
  } catch (error) {
    console.error("Get breaks error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
}

module.exports = getBreaks;

