const { query } = require("./shared");
const { recalculateAttendanceHours } = require("./breakHelpers");

async function updateBreak(req, res) {
  try {
    const { break_id } = req.params;
    const { start_time, end_time } = req.body;

    if (!start_time) {
      return res.status(400).json({ error: "Start time is required" });
    }

    // Verify break belongs to admin
    const breaks = await query(
      `SELECT ws.id, ws.attendance_id, ws.staff_id
      FROM work_sessions ws
      JOIN attendance a ON ws.attendance_id = a.id
      JOIN staff s ON a.staff_id = s.id
      JOIN branches b ON s.branch_id = b.id
      JOIN shops sh ON b.shop_id = sh.id
      WHERE ws.id = ? AND sh.admin_id = ? AND ws.session_type = 'break'`,
      [break_id, req.user.id]
    );

    if (!breaks || breaks.length === 0) {
      return res
        .status(403)
        .json({ error: "Access denied or break not found" });
    }

    const attendanceId = breaks[0].attendance_id;
    const startDateTime = new Date(start_time);
    let endDateTime = end_time ? new Date(end_time) : null;
    let durationMinutes = null;

    if (endDateTime) {
      if (endDateTime < startDateTime) {
        return res
          .status(400)
          .json({ error: "End time must be after start time" });
      }
      durationMinutes = Math.round(
        (endDateTime - startDateTime) / (1000 * 60)
      );
    }

    await query(
      `UPDATE work_sessions
      SET start_time = ?, end_time = ?, duration_minutes = ?
      WHERE id = ?`,
      [startDateTime, endDateTime, durationMinutes, break_id]
    );

    // Recalculate total hours for the attendance record
    await recalculateAttendanceHours(attendanceId);

    res.json({
      message: "Break updated successfully",
      break_session: {
        id: break_id,
        start_time: startDateTime.toISOString(),
        end_time: endDateTime?.toISOString() || null,
        duration_minutes: durationMinutes,
      },
    });
  } catch (error) {
    console.error("Update break error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
}

module.exports = updateBreak;

