const AppError = require("../../../utils/appError");
const conn = require("../../../services/db");
const crypto = require("crypto");
const { sendPasswordResetEmail } = require("../../../utils/emailService");
const getEnvConfig = require("../../../utils/Config");

async function forgotPassword(req, res, next) {
  const { email } = req?.body;

  try {
    // Validate required fields
    if (!email) {
      return res.status(400).json({
        status: "error",
        message: "Email is required",
      });
    }

    // Trim and lowercase email
    const trimmedEmail = email?.trim()?.toLowerCase();

    // Validate email format
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(trimmedEmail)) {
      return res.status(400).json({
        status: "error",
        message: "Please enter a valid email address",
      });
    }

    // Find user by email
    const query =
      "SELECT id, email, first_name, last_name, full_name, password FROM `users` WHERE `is_deleted` IS NULL AND email = ?";
    
    conn.query(
      query,
      [trimmedEmail],
      async function (err, result, fields) {
        if (err) {
          const errorMessage =
            err?.message || "Something went wrong, Please try again";
          return next(new AppError(errorMessage, 500));
        }

        // Always return success message for security (don't reveal if email exists)
        // But only send email if user exists and has a password
        if (result?.length > 0) {
          const user = result[0];

          // Only send reset email if user has a password (not OAuth-only users)
          if (user.password) {
            // Generate password reset token
            const resetToken = crypto.randomBytes(32).toString("hex");
            const resetTokenExpiry = new Date(Date.now() + 60 * 60 * 1000); // 1 hour from now

            // Update user with reset token
            const updateQuery =
              "UPDATE `users` SET password_reset_token = ?, password_reset_token_expiry = ? WHERE id = ?";
            
            conn.query(
              updateQuery,
              [resetToken, resetTokenExpiry, user.id],
              async function (updateErr, updateResult) {
                if (updateErr) {
                  console.error("Error updating password reset token:", updateErr);
                  // Still return success for security
                  return res.status(200).json({
                    status: "success",
                    message: "If an account with that email exists, a password reset link has been sent.",
                  });
                }

                // Send password reset email
                const baseUrl = getEnvConfig("FRONTEND_URL") || getEnvConfig("APP_URL") || "http://localhost:3000";
                const emailResult = await sendPasswordResetEmail(
                  trimmedEmail,
                  user.full_name || `${user.first_name} ${user.last_name}`.trim() || "User",
                  resetToken,
                  baseUrl
                );

                if (!emailResult.success) {
                  console.error("Failed to send password reset email:", emailResult.error);
                  // Still return success for security
                }

                return res.status(200).json({
                  status: "success",
                  message: "If an account with that email exists, a password reset link has been sent.",
                });
              }
            );
          } else {
            // User exists but has no password (OAuth-only user)
            // Return success for security (don't reveal account exists)
            return res.status(200).json({
              status: "success",
              message: "If an account with that email exists, a password reset link has been sent.",
            });
          }
        } else {
          // User doesn't exist - return success for security
          return res.status(200).json({
            status: "success",
            message: "If an account with that email exists, a password reset link has been sent.",
          });
        }
      }
    );
  } catch (e) {
    console.log("Exception Error : Forgot Password ", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
}

module.exports = forgotPassword;

