const AppError = require("../../../utils/appError");
const conn = require("../../../services/db");
const DbHelper = require("../../../helpers/DbHelper");
const { Compare } = require("../../../utils/encryption");
const moment = require("moment");

async function login(req, res, next) {
  const { email, password } = req?.body;

  try {
    // Validate required fields
    if (!email || !password) {
      if (!email) {
        return res.status(400).json({
          status: "error",
          message: "Email is required",
          errorType: "email"
        });
      }
      if (!password) {
        return res.status(400).json({
          status: "error",
          message: "Password is required",
          errorType: "password"
        });
      }
    }

    // Trim and lowercase email
    const trimmedEmail = email?.trim()?.toLowerCase();

    // Validate email format
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(trimmedEmail)) {
      return res.status(400).json({
        status: "error",
        message: "Please enter a valid email address",
        errorType: "email"
      });
    }

    // Find user by email
    const query =
      "SELECT * FROM `users` WHERE `is_deleted` IS NULL AND email = ?";
    
    conn.query(
      query,
      [trimmedEmail],
      async function (err, result, fields) {
        if (err) {
          const errorMessage =
            err?.message || "Something went wrong, Please try again";
          return next(new AppError(errorMessage, 500));
        }

        if (result?.length === 0) {
          // Email doesn't exist - email error
          return res.status(401).json({
            status: "error",
            message: "Invalid email address",
            errorType: "email"
          });
        }

        const user = result[0];

        // Check if user has a password (for OAuth-only users)
        if (!user.password) {
          return res.status(401).json({
            status: "error",
            message: "Please use social login for this account",
            errorType: "email"
          });
        }

        // Check if email is verified
        if (user.email_verified !== 1 && user.email_verified !== true) {
          return res.status(401).json({
            status: "error",
            message: "Please verify your email address before logging in. Check your inbox for the verification email.",
            errorType: "email",
            requiresVerification: true
          });
        }

        // Verify password
        const isPasswordCorrect = await Compare(password, user.password);
        
        if (!isPasswordCorrect) {
          // Password is incorrect - password error
          return res.status(401).json({
            status: "error",
            message: "Invalid password",
            errorType: "password"
          });
        }

        // Generate token (expires in 1 year)
        const tokenExpireDate = moment().add(1, "year").valueOf();
        const accessToken = `${user.id}_${Date.now()}_${Math.random().toString(36).substring(7)}`;

        // Insert new session into user_sessions table (supports multiple devices)
        const insertSessionQuery =
          "INSERT INTO `user_sessions` (`user_id`, `token`, `expires_at`) VALUES (?, ?, ?)";
        
        conn.query(
          insertSessionQuery,
          [user.id, accessToken, tokenExpireDate],
          async function (updateErr, updateResult) {
            if (updateErr) {
              return next(new AppError("Failed to create login session", 500));
            }

            // Check for active subscription
            const subscriptionQuery = `
              SELECT 
                us.id as subscription_id,
                us.plan_type,
                us.amount,
                us.start_date,
                us.end_date,
                us.status as subscription_status,
                us.created_at as subscription_created_at,
                sp.id as plan_id,
                sp.name as plan_name,
                sp.price as plan_price,
                sp.duration_type,
                sp.max_locations,
                sp.max_staff,
                sp.description as plan_description
              FROM user_subscriptions us
              LEFT JOIN subscription_plans sp ON us.subscription_plan_id = sp.id
              WHERE us.user_id = ? 
                AND us.status = 'active' 
                AND us.end_date > NOW()
              ORDER BY us.created_at DESC
              LIMIT 1
            `;

            try {
              const subscriptionResult = await DbHelper.promisifyQuery(subscriptionQuery, conn, next, [user.id]);
              
              const hasSubscription = subscriptionResult && subscriptionResult.length > 0;
              const subscriptionData = hasSubscription ? subscriptionResult[0] : null;

              // Fetch shops for the logged-in user
              const shopsQuery = `
                SELECT 
                  id,
                  name,
                  admin_id,
                  address,
                  contact_phone,
                  is_active,
                  created_at,
                  updated_at
                FROM shops
                WHERE admin_id = ?
                ORDER BY created_at DESC
              `;

              let shopsData = [];
              try {
                const shopsResult = await DbHelper.promisifyQuery(shopsQuery, conn, next, [user.id]);
                shopsData = shopsResult || [];
              } catch (shopsErr) {
                console.log("Error fetching shops:", shopsErr);
                // Continue even if shops fetch fails
                shopsData = [];
              }

              return res.status(200).json({
                status: "success",
                message: "Logged in successfully.",
                data: {
                  accessToken,
                  user: {
                    id: user.id,
                    firstName: user.first_name,
                    lastName: user.last_name,
                    fullName: user.full_name,
                    role: user.role,
                    email: user.email,
                    shopName: user.shop_name,
                    location: user.location,
                  },
                  hasSubscription: hasSubscription,
                  subscription: subscriptionData,
                  shops: shopsData,
                },
              });
            } catch (subErr) {
              console.log("Error fetching subscription:", subErr);
              
              // Fetch shops even if subscription fetch fails
              const shopsQuery = `
                SELECT 
                  id,
                  name,
                  admin_id,
                  address,
                  contact_phone,
                  is_active,
                  created_at,
                  updated_at
                FROM shops
                WHERE admin_id = ?
                ORDER BY created_at DESC
              `;

              let shopsData = [];
              try {
                const shopsResult = await DbHelper.promisifyQuery(shopsQuery, conn, next, [user.id]);
                shopsData = shopsResult || [];
              } catch (shopsErr) {
                console.log("Error fetching shops:", shopsErr);
                shopsData = [];
              }

              // Return login success even if subscription fetch fails
              return res.status(200).json({
                status: "success",
                message: "Logged in successfully.",
                data: {
                  accessToken,
                  user: {
                    id: user.id,
                    firstName: user.first_name,
                    lastName: user.last_name,
                    fullName: user.full_name,
                    role: user.role,
                    email: user.email,
                    shopName: user.shop_name,
                    location: user.location,
                  },
                  hasSubscription: false,
                  subscription: null,
                  shops: shopsData,
                },
              });
            }
          }
        );
      }
    );
  } catch (e) {
    console.log("Exception Error : Frontend User Login ", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
}

module.exports = login;
