const AppError = require("../../../utils/appError");
const conn = require("../../../services/db");
const { Hash } = require("../../../utils/encryption");

async function resetPassword(req, res, next) {
  // Get token from query params (GET request) or body (POST request)
  let token = req?.query?.token || req?.body?.token;
  const { password, confirmPassword } = req?.body;

  // Trim and decode the token in case of URL encoding issues
  if (token) {
    token = token.trim();
    // Decode URI component in case it was double-encoded
    try {
      token = decodeURIComponent(token);
    } catch (e) {
      // If decoding fails, use original token
      console.log("Token decode warning:", e.message);
    }
  }

  try {
    // Validate token
    if (!token) {
      return res.status(400).json({
        status: "error",
        message: "Reset token is required",
      });
    }

    // Validate password fields
    if (!password || !confirmPassword) {
      return res.status(400).json({
        status: "error",
        message: "Password and confirm password are required",
      });
    }

    // Validate password match
    if (password !== confirmPassword) {
      return res.status(400).json({
        status: "error",
        message: "Password and confirm password do not match",
      });
    }

    // Validate password length (minimum 6 characters)
    if (password.length < 6) {
      return res.status(400).json({
        status: "error",
        message: "Password must be at least 6 characters long",
      });
    }

    // Find user by reset token
    const query =
      "SELECT id, email, password_reset_token_expiry FROM `users` WHERE `is_deleted` IS NULL AND password_reset_token = ?";
    
    conn.query(
      query,
      [token],
      async function (err, result, fields) {
        if (err) {
          const errorMessage =
            err?.message || "Something went wrong, Please try again";
          return next(new AppError(errorMessage, 500));
        }

        if (!result || result.length === 0) {
          return res.status(400).json({
            status: "error",
            message: "Invalid or expired reset token. Please request a new password reset link.",
          });
        }

        const user = result[0];

        // Check if token has expired
        const now = new Date();
        const expiryDate = new Date(user.password_reset_token_expiry);
        if (expiryDate < now) {
          return res.status(400).json({
            status: "error",
            message: "Reset token has expired. Please request a new password reset link.",
          });
        }

        // Hash the new password
        const hashedPassword = await Hash(password);

        // Update user password and clear reset token
        const updateQuery =
          "UPDATE `users` SET password = ?, password_reset_token = NULL, password_reset_token_expiry = NULL WHERE id = ? AND password_reset_token = ?";
        
        conn.query(
          updateQuery,
          [hashedPassword, user.id, token],
          async function (updateErr, updateResult) {
            if (updateErr) {
              const errorMessage =
                updateErr?.message || "Something went wrong, Please try again";
              return next(new AppError(errorMessage, 500));
            }

            if (updateResult && updateResult.affectedRows > 0) {
              return res.status(200).json({
                status: "success",
                message: "Your password has been reset successfully. You can now log in with your new password.",
              });
            } else {
              return res.status(400).json({
                status: "error",
                message: "Failed to reset password. Please try again or request a new reset link.",
              });
            }
          }
        );
      }
    );
  } catch (e) {
    console.log("Exception Error : Reset Password ", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
}

module.exports = resetPassword;

