const AppError = require("../../../utils/appError");
const conn = require("../../../services/db");
const { Hash } = require("../../../utils/encryption");
const crypto = require("crypto");
const { sendVerificationEmail } = require("../../../utils/emailService");
const getEnvConfig = require("../../../utils/Config");

async function userRegistration(req, res, next) {
  const {
    firstName,
    lastName,
    email,
    phone,
    shopName,
    location,
    password,
    confirmPassword,
  } = req?.body;

  try {
    // Trim and validate required fields
    const trimmedFirstName = firstName?.trim();
    const trimmedLastName = lastName?.trim();
    const trimmedEmail = email?.trim()?.toLowerCase();
    const trimmedPhone = phone?.trim();
    const trimmedShopName = shopName?.trim();
    const trimmedLocation = location?.trim();
    const trimmedPassword = password?.trim();
    const trimmedConfirmPassword = confirmPassword?.trim();

    // Validate required fields with better error messages
    if (!trimmedFirstName) {
      return next(new AppError("First name is required", 400));
    }
    if (!trimmedLastName) {
      return next(new AppError("Last name is required", 400));
    }
    if (!trimmedEmail) {
      return next(new AppError("Email is required", 400));
    }
    // Validate email format
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(trimmedEmail)) {
      return next(new AppError("Please enter a valid email address", 400));
    }
    if (!trimmedShopName) {
      return next(new AppError("Business is required", 400));
    }
    if (!trimmedPhone) {
      return next(new AppError("Phone is required", 400));
    }
    if (!trimmedLocation) {
      return next(new AppError("Location is required", 400));
    }
    if (!trimmedPassword) {
      return next(new AppError("Password is required", 400));
    }
    if (!trimmedConfirmPassword) {
      return next(new AppError("Confirm password is required", 400));
    }

    // Validate password length (minimum 6 characters)
    if (trimmedPassword.length < 6) {
      return next(new AppError("Password must be at least 6 characters long", 400));
    }

    // Validate password match
    if (trimmedPassword !== trimmedConfirmPassword) {
      return next(new AppError("Password and confirm password do not match", 400));
    }

    // Check if email already exists
    const checkEmailQuery = "SELECT id FROM `users` WHERE `is_deleted` IS NULL AND email = ?";
    conn.query(
      checkEmailQuery,
      [trimmedEmail],
      async function (err, result, fields) {
        if (err) {
          return next(new AppError("Something went wrong, Please try again", 500));
        }
        
        if (result && result.length > 0) {
          return next(new AppError("Email already exists. Please use a different email.", 400));
        }

        // Proceed with registration if email is unique
        // Construct full_name from firstName and lastName
        const full_name = `${trimmedFirstName} ${trimmedLastName}`.trim();

        // Hash the password
        const hashedPassword = await Hash(trimmedPassword);

        // Generate verification token
        const verificationToken = crypto.randomBytes(32).toString("hex");
        const verificationTokenExpiry = new Date(Date.now() + 24 * 60 * 60 * 1000); // 24 hours from now

        // Insert user into database with email_verified = 0 and verification_token
        const query =
          "INSERT INTO `users` (first_name, last_name, full_name, email, phone, shop_name, location, password, email_verified, verification_token, verification_token_expiry) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        conn.query(
          query,
          [trimmedFirstName, trimmedLastName, full_name, trimmedEmail, trimmedPhone, trimmedShopName, trimmedLocation, hashedPassword, 0, verificationToken, verificationTokenExpiry],
          async function (err, result, fields) {
            if (err) {
              // Check if error is due to duplicate email (in case of race condition)
              // MySQL error code 1062 is ER_DUP_ENTRY
              const isDuplicateEntry = 
                err.code === 'ER_DUP_ENTRY' || 
                err.code === 1062 ||
                err.errno === 1062 ||
                (err.message && (
                  err.message.includes('Duplicate entry') || 
                  err.message.includes('Duplicate') ||
                  err.message.toLowerCase().includes('email')
                ));
              
              if (isDuplicateEntry) {
                return next(new AppError("Email already exists. Please use a different email.", 400));
              }
              const errorMessage =
                err?.message || "Something went wrong, Please try again";
              return next(new AppError(errorMessage, 500));
            } else if (result) {
              // Send verification email
              const baseUrl = getEnvConfig("FRONTEND_URL") || getEnvConfig("APP_URL") || "http://localhost:3000";
              const emailResult = await sendVerificationEmail(
                trimmedEmail,
                full_name,
                verificationToken,
                baseUrl
              );

              if (!emailResult.success) {
                console.error("Failed to send verification email:", emailResult.error);
                // Still return success but log the error
                // User can request resend later
              }

              return res.status(200).json({
                status: "success",
                message: "Registration successful! Please check your email to verify your account.",
              });
            }
          }
        );
      }
    );
  } catch (e) {
    console.log("Exception Error : Frontend User Registration ", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
}

module.exports = userRegistration;
