const AppError = require("../../../utils/appError");
const conn = require("../../../services/db");

async function verifyEmail(req, res, next) {
  // Get token from query params (GET request) or body (POST request)
  let token = req?.query?.token || req?.body?.token;
  
  // Trim and decode the token in case of URL encoding issues
  if (token) {
    token = token.trim();
    // Decode URI component in case it was double-encoded
    try {
      token = decodeURIComponent(token);
    } catch (e) {
      // If decoding fails, use original token
      console.log("Token decode warning:", e.message);
    }
  }

  // Debug logging (remove in production)
  console.log("Verification request received:", {
    method: req.method,
    query: req.query,
    body: req.body,
    tokenLength: token ? token.length : 0,
    tokenPreview: token ? token.substring(0, 20) + "..." : "missing",
  });

  try {
    if (!token) {
      console.log("Token missing in request");
      return res.status(400).json({
        status: "error",
        message: "Verification token is required",
      });
    }

    // Find user by verification token
    const query =
      "SELECT id, email, email_verified, verification_token_expiry FROM `users` WHERE `is_deleted` IS NULL AND verification_token = ?";
    
    conn.query(
      query,
      [token],
      async function (err, result, fields) {
        if (err) {
          const errorMessage =
            err?.message || "Something went wrong, Please try again";
          return next(new AppError(errorMessage, 500));
        }

        if (!result || result.length === 0) {
          // Token not found - token was cleared after verification (email already verified)
          // This is a second-time verification attempt
          return res.status(200).json({
            status: "success",
            message: "Email already verified. You can now log in.",
          });
        }

        const user = result[0];

        // Debug: Log the actual values
        console.log("Verification check - User data:", {
          userId: user.id,
          email: user.email,
          email_verified_raw: user.email_verified,
          email_verified_type: typeof user.email_verified,
          email_verified_number: Number(user.email_verified),
        });

        // Check if email already verified
        // Convert to number for comparison (MySQL TINYINT can return as number or string)
        const emailVerifiedValue = Number(user.email_verified);
        const isVerified = emailVerifiedValue === 1;
        
        if (isVerified) {
          // Email already verified - second time or later
          console.log("Email already verified, skipping update");
          return res.status(200).json({
            status: "success",
            message: "Email already verified. You can now log in.",
          });
        }
        
        console.log("Email not verified yet, proceeding with verification. Current value:", emailVerifiedValue);

        // Check if token has expired
        const now = new Date();
        const expiryDate = new Date(user.verification_token_expiry);
        if (expiryDate < now) {
          return res.status(400).json({
            status: "error",
            message: "Verification token has expired. Please request a new verification email.",
          });
        }

        // Update user to mark email as verified and clear verification token
        const updateQuery =
          "UPDATE `users` SET email_verified = 1, verification_token = NULL, verification_token_expiry = NULL WHERE id = ? AND verification_token = ?";
        
        conn.query(
          updateQuery,
          [user.id, token],
          async function (updateErr, updateResult) {
            if (updateErr) {
              const errorMessage =
                updateErr?.message || "Something went wrong, Please try again";
              return next(new AppError(errorMessage, 500));
            }

            // Check if update was successful
            console.log("Update query result:", {
              affectedRows: updateResult?.affectedRows,
              changedRows: updateResult?.changedRows,
            });
            
            if (updateResult && updateResult.affectedRows > 0) {
              // First time verification - return success message
              console.log("Email verified successfully for the first time");
              return res.status(200).json({
                status: "success",
                message: "Your email address was verified successfully.",
              });
            } else {
              console.log("Update did not affect any rows - checking current status");
              // Update didn't affect any rows - might be a race condition
              // Check if email is now verified (already verified by another request)
              const checkQuery = "SELECT email_verified FROM `users` WHERE id = ?";
              conn.query(checkQuery, [user.id], function(checkErr, checkResult) {
                if (!checkErr && checkResult && checkResult.length > 0) {
                  if (checkResult[0].email_verified === 1) {
                    // Already verified - return appropriate message
                    return res.status(200).json({
                      status: "success",
                      message: "Email is verified. You can now log in...",
                    });
                  }
                }
                return res.status(400).json({
                  status: "error",
                  message: "Verification failed. Please try again or contact support.",
                });
              });
            }
          }
        );
      }
    );
  } catch (e) {
    console.log("Exception Error : Email Verification ", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
}

module.exports = verifyEmail;

