const { query } = require("./shared");

// Public endpoint: check if a feature is enabled
async function checkFeatureFlag(req, res) {
  try {
    // Check if shop_id column exists
    const columns = await query(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'feature_flags' 
      AND COLUMN_NAME = 'shop_id'
    `);

    const hasShopIdColumn = columns.length > 0;
    const { featureName } = req.params;
    const shopId = req.query.shop_id;

    // If shopId is provided and column exists, check for shop-specific flag first
    if (hasShopIdColumn && shopId) {
      const shopFlags = await query(
        "SELECT is_enabled FROM feature_flags WHERE feature_name = ? AND shop_id = ?",
        [featureName, shopId],
      );

      if (shopFlags.length > 0) {
        return res.json({
          feature_name: featureName,
          is_enabled:
            shopFlags[0].is_enabled === 1 || shopFlags[0].is_enabled === true,
          source: "shop",
        });
      }

      // Fallback to global flag
      const globalFlags = await query(
        "SELECT is_enabled FROM feature_flags WHERE feature_name = ? AND shop_id IS NULL",
        [featureName],
      );

      if (globalFlags.length > 0) {
        return res.json({
          feature_name: featureName,
          is_enabled:
            globalFlags[0].is_enabled === 1 ||
            globalFlags[0].is_enabled === true,
          source: "global",
        });
      }
    } else {
      // Fallback: old behavior without shop_id
      const flags = await query(
        "SELECT is_enabled FROM feature_flags WHERE feature_name = ?",
        [featureName],
      );

      if (flags.length > 0) {
        return res.json({
          feature_name: featureName,
          is_enabled: flags[0].is_enabled === 1 || flags[0].is_enabled === true,
          source: "global",
        });
      }
    }

    // Default to enabled if flag doesn't exist at all
    return res.json({
      feature_name: featureName,
      is_enabled: true,
      source: "default",
    });
  } catch (error) {
    console.error("Check feature flag error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
}

module.exports = checkFeatureFlag;


