const { query } = require("./shared");

// Get a specific feature flag (superadmin only)
async function getFeatureFlag(req, res) {
  try {
    if (req.user.role !== "superadmin") {
      return res
        .status(403)
        .json({ error: "Access denied. Super admin only." });
    }

    // Check if shop_id column exists
    const columns = await query(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'feature_flags' 
      AND COLUMN_NAME = 'shop_id'
    `);

    const hasShopIdColumn = columns.length > 0;
    const { featureName } = req.params;
    const shopId = req.query.shop_id;

    let query;
    const params = [featureName];

    if (hasShopIdColumn) {
      query =
        "SELECT feature_name, is_enabled, updated_at FROM feature_flags WHERE feature_name = ? AND shop_id";
      if (shopId) {
        query += " = ?";
        params.push(shopId);
      } else {
        query += " IS NULL";
      }
    } else {
      // Fallback: old behavior without shop_id
      query =
        "SELECT feature_name, is_enabled, updated_at FROM feature_flags WHERE feature_name = ?";
    }

    const flags = await query(query, params);

    if (flags.length === 0) {
      return res.status(404).json({ error: "Feature flag not found" });
    }

    const flag = flags[0];
    res.json({
      feature_name: flag.feature_name,
      is_enabled: flag.is_enabled === 1 || flag.is_enabled === true,
      updated_at: flag.updated_at,
    });
  } catch (error) {
    console.error("Get feature flag error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
}

module.exports = getFeatureFlag;


