const { query } = require("./shared");

// Get all feature flags (superadmin only)
async function getFeatureFlags(req, res) {
  try {
    if (req.user.role !== "superadmin") {
      return res
        .status(403)
        .json({ error: "Access denied. Super admin only." });
    }

    // Check if shop_id column exists
    const columns = await query(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'feature_flags' 
      AND COLUMN_NAME = 'shop_id'
    `);

    const hasShopIdColumn = columns.length > 0;
    const shopId = req.query.shop_id;

    let query;
    const params = [];

    if (hasShopIdColumn) {
      query =
        "SELECT feature_name, is_enabled, shop_id, updated_at FROM feature_flags";
      if (shopId) {
        query += " WHERE shop_id = ?";
        params.push(shopId);
      } else {
        query += " WHERE shop_id IS NULL";
      }
    } else {
      // Fallback: old behavior without shop_id
      query = "SELECT feature_name, is_enabled, updated_at FROM feature_flags";
    }

    query += " ORDER BY feature_name";

    const flags = await query(query, params);

    // Convert to object format for easier access
    const flagsObject = {};
    flags.forEach((flag) => {
      flagsObject[flag.feature_name] = {
        is_enabled: flag.is_enabled === 1 || flag.is_enabled === true,
        updated_at: flag.updated_at,
        shop_id: hasShopIdColumn ? flag.shop_id || null : null,
      };
    });

    res.json({
      message: "Feature flags retrieved successfully",
      feature_flags: flagsObject,
      shop_id: hasShopIdColumn ? shopId || null : null,
    });
  } catch (error) {
    console.error("Get feature flags error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
}

module.exports = getFeatureFlags;


