const { query } = require("./shared");

// Update feature flag (superadmin only)
async function updateFeatureFlag(req, res) {
  try {
    if (req.user.role !== "superadmin") {
      return res
        .status(403)
        .json({ error: "Access denied. Super admin only." });
    }

    const { featureName } = req.params;
    const { is_enabled, shop_id } = req.body;

    if (typeof is_enabled !== "boolean") {
      return res
        .status(400)
        .json({ error: "is_enabled must be a boolean value" });
    }

    // Check if shop_id column exists in feature_flags table
    const columns = await query(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'feature_flags' 
      AND COLUMN_NAME = 'shop_id'
    `);

    const hasShopIdColumn = columns.length > 0;

    if (hasShopIdColumn) {
      // New behavior: support shop-specific flags
      let checkQuery =
        "SELECT id FROM feature_flags WHERE feature_name = ? AND shop_id";
      const checkParams = [featureName];

      if (shop_id) {
        checkQuery += " = ?";
        checkParams.push(shop_id);
      } else {
        checkQuery += " IS NULL";
      }

      const existing = await query(checkQuery, checkParams);

      if (existing.length === 0) {
        // Create if doesn't exist
        await query(
          "INSERT INTO feature_flags (feature_name, is_enabled, shop_id) VALUES (?, ?, ?)",
          [featureName, is_enabled, shop_id || null],
        );
      } else {
        // Update existing
        let updateQuery =
          "UPDATE feature_flags SET is_enabled = ? WHERE feature_name = ? AND shop_id";
        const updateParams = [is_enabled, featureName];

        if (shop_id) {
          updateQuery += " = ?";
          updateParams.push(shop_id);
        } else {
          updateQuery += " IS NULL";
        }

        await query(updateQuery, updateParams);
      }
    } else {
      // Fallback: old behavior without shop_id (global flags only)
      const existing = await query(
        "SELECT id FROM feature_flags WHERE feature_name = ?",
        [featureName],
      );

      if (existing.length === 0) {
        // Create if doesn't exist
        await query(
          "INSERT INTO feature_flags (feature_name, is_enabled) VALUES (?, ?)",
          [featureName, is_enabled],
        );
      } else {
        // Update existing
        await query(
          "UPDATE feature_flags SET is_enabled = ? WHERE feature_name = ?",
          [is_enabled, featureName],
        );
      }
    }

    res.json({
      message: "Feature flag updated successfully",
      feature_name: featureName,
      is_enabled,
      shop_id: hasShopIdColumn ? shop_id || null : null,
    });
  } catch (error) {
    console.error("Update feature flag error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
}

module.exports = updateFeatureFlag;


