const AppError = require("../../../utils/appError");
const conn = require("../../../services/db");
const DbHelper = require("../../../helpers/DbHelper");

async function profileInfo(req, res, next) {
  try {
    const { user_id } = req.headers;

    if (!user_id) {
      return res.status(200).json({
        status: "error",
        message: "User authentication required",
      });
    }

    // Fetch user profile information
    const userQuery = `
      SELECT 
        id,
        first_name,
        last_name,
        CONCAT(first_name, ' ', last_name) as full_name,
        email,
        phone,
        shop_name,
        location,
        profile_picture,
        provider
      FROM users 
      WHERE id = ? AND is_deleted IS NULL
    `;

    const userResult = await DbHelper.promisifyQuery(userQuery, conn, next, [user_id]);

    if (!userResult || userResult.length === 0) {
      return res.status(200).json({
        status: "error",
        message: "User not found",
      });
    }

    // Fetch active subscription information (including cancelled but not expired)
    const subscriptionQuery = `
      SELECT 
        us.id as subscription_id,
        us.plan_type,
        us.amount,
        us.start_date,
        us.end_date,
        us.status as subscription_status,
        us.stripe_subscription_id,
        us.created_at as subscription_created_at,
        sp.id as plan_id,
        sp.name as plan_name,
        sp.price as plan_price,
        sp.duration_type,
        sp.max_locations,
        sp.max_staff,
        sp.description as plan_description
      FROM user_subscriptions us
      LEFT JOIN subscription_plans sp ON us.subscription_plan_id = sp.id
      WHERE us.user_id = ? 
        AND (us.status = 'active' OR (us.status = 'cancelled' AND us.end_date > NOW()))
        AND us.end_date > NOW()
      ORDER BY us.created_at DESC
      LIMIT 1
    `;

    const subscriptionResult = await DbHelper.promisifyQuery(subscriptionQuery, conn, next, [user_id]);

    // Convert subscription amount and plan_price from string to number (MySQL DECIMAL returns as string)
    let subscription = null;
    if (subscriptionResult && subscriptionResult.length > 0) {
      subscription = {
        ...subscriptionResult[0],
        amount: parseFloat(subscriptionResult[0].amount) || 0,
        plan_price: subscriptionResult[0].plan_price ? parseFloat(subscriptionResult[0].plan_price) || 0 : null,
      };
    }

    // Combine user data with subscription data
    const userData = {
      ...userResult[0],
      subscription: subscription,
    };

    res.status(200).json({
      status: "success",
      message: "Profile information fetched successfully",
      data: userData,
    });
  } catch (e) {
    console.log("Exception Error : Profile Info ", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
}

module.exports = profileInfo;
