const {
  query,
  ensureBranchAccess,
  getBreakTypeMap,
} = require("./shared");

async function getAssignments(req, res) {
  try {
    const branchId = Number(req.params.branch_id);
    if (Number.isNaN(branchId)) {
      return res.status(400).json({ error: "Invalid branch ID" });
    }

    const { start_date: startDate, end_date: endDate } = req.query;

    await ensureBranchAccess(branchId, req.user.id);

    let assignmentsQuery = `
      SELECT 
        ra.id,
        ra.branch_id,
        DATE_FORMAT(ra.shift_date, '%Y-%m-%d') AS shift_date,
        ra.staff_id,
        ra.shift_type_id,
        ra.position_id,
        ra.start_minutes,
        ra.end_minutes,
        ra.break_type_ids,
        ra.notes,
        ra.created_by_type,
        ra.created_by_id,
        ra.created_at,
        ra.updated_at,
        st.full_name,
        st.staff_code,
        st.phone,
        st.position,
        rst.name AS shift_type_name,
        rst.start_minutes AS default_start_minutes,
        rst.end_minutes AS default_end_minutes,
        rp.name AS position_name
      FROM roster_assignments ra
      JOIN staff st ON ra.staff_id = st.id
      JOIN roster_shift_types rst ON ra.shift_type_id = rst.id
      LEFT JOIN roster_positions rp ON ra.position_id = rp.id
      WHERE ra.branch_id = ?
    `;
    const params = [branchId];

    if (startDate) {
      assignmentsQuery += " AND ra.shift_date >= ?";
      params.push(startDate);
    }

    if (endDate) {
      assignmentsQuery += " AND ra.shift_date <= ?";
      params.push(endDate);
    }

    assignmentsQuery += " ORDER BY ra.shift_date ASC, rst.start_minutes ASC";

    const rows = await query(assignmentsQuery, params);
    const breakTypes = await query(
      "SELECT * FROM roster_break_types WHERE branch_id = ?",
      [branchId]
    );
    const breakMap = getBreakTypeMap(breakTypes);

    const assignments = rows.map((row) => {
      let breakIds = [];
      if (row.break_type_ids) {
        try {
          const parsed =
            typeof row.break_type_ids === "string"
              ? JSON.parse(row.break_type_ids)
              : row.break_type_ids;
          if (Array.isArray(parsed)) {
            breakIds = parsed;
          }
        } catch (parseError) {
          console.warn("Failed to parse break IDs for assignment", row.id);
        }
      }

      const breaks = breakIds.map((id) => breakMap.get(id)).filter(Boolean);

      return {
        id: row.id,
        branch_id: row.branch_id,
        shift_date: row.shift_date,
        staff: {
          id: row.staff_id,
          full_name: row.full_name,
          staff_code: row.staff_code,
          phone: row.phone,
          position: row.position,
        },
        shift_type: {
          id: row.shift_type_id,
          name: row.shift_type_name,
          startMinutes: row.start_minutes ?? row.default_start_minutes,
          endMinutes: row.end_minutes ?? row.default_end_minutes,
        },
        position: row.position_id
          ? {
              id: row.position_id,
              name: row.position_name,
            }
          : null,
        start_minutes: row.start_minutes,
        end_minutes: row.end_minutes,
        breaks,
        notes: row.notes,
      };
    });

    res.json({ assignments });
  } catch (error) {
    console.error("Get roster assignments error:", error);
    res.status(error.status || 500).json({
      error: error.message || "Failed to fetch roster assignments",
    });
  }
}

module.exports = getAssignments;

