const util = require("util");
const conn = require("../../../services/db");

const query = util.promisify(conn.query).bind(conn);
const beginTransaction = util.promisify(conn.beginTransaction).bind(conn);
const commit = util.promisify(conn.commit).bind(conn);
const rollback = util.promisify(conn.rollback).bind(conn);

const ensureBranchAccess = async (branchId, adminId) => {
  const rows = await query(
    `
    SELECT 
      b.id,
      b.name,
      b.shop_id,
      s.name AS shop_name,
      s.admin_id
    FROM branches b
    JOIN shops s ON b.shop_id = s.id
    WHERE b.id = ?
  `,
    [branchId]
  );

  if (!rows || rows.length === 0) {
    const error = new Error("Branch not found");
    error.status = 404;
    throw error;
  }

  if (rows[0].admin_id !== adminId) {
    const error = new Error("You do not have access to this branch");
    error.status = 403;
    throw error;
  }

  return rows[0];
};

const ensureRosterEntityAccess = async (table, id, adminId) => {
  const rows = await query(
    `
    SELECT 
      entity.*,
      b.id AS branch_id,
      s.admin_id
    FROM ${table} entity
    JOIN branches b ON entity.branch_id = b.id
    JOIN shops s ON b.shop_id = s.id
    WHERE entity.id = ?
  `,
    [id]
  );

  if (!rows || rows.length === 0) {
    const error = new Error("Item not found");
    error.status = 404;
    throw error;
  }

  if (rows[0].admin_id !== adminId) {
    const error = new Error("You do not have permission to modify this item");
    error.status = 403;
    throw error;
  }

  return rows[0];
};

const formatShiftType = (row) => ({
  id: row.id,
  branch_id: row.branch_id,
  name: row.name,
  startMinutes: row.start_minutes,
  endMinutes: row.end_minutes,
  created_at: row.created_at,
  updated_at: row.updated_at,
});

const formatPosition = (row) => ({
  id: row.id,
  branch_id: row.branch_id,
  name: row.name,
  created_at: row.created_at,
  updated_at: row.updated_at,
});

const formatBreakType = (row) => ({
  id: row.id,
  branch_id: row.branch_id,
  name: row.name,
  durationMinutes: row.duration_minutes,
  created_at: row.created_at,
  updated_at: row.updated_at,
});

const getBreakTypeMap = (breakTypes) => {
  const map = new Map();
  (breakTypes || []).forEach((item) => {
    map.set(item.id, formatBreakType(item));
  });
  return map;
};

module.exports = {
  query,
  beginTransaction,
  commit,
  rollback,
  ensureBranchAccess,
  ensureRosterEntityAccess,
  formatShiftType,
  formatPosition,
  formatBreakType,
  getBreakTypeMap,
};

