const {
  query,
  beginTransaction,
  commit,
  rollback,
  ensureBranchAccess,
  ensureRosterEntityAccess,
  getBreakTypeMap,
} = require("./shared");

async function upsertAssignments(req, res) {
  let transactionStarted = false;

  const rollbackIfNeeded = async () => {
    if (transactionStarted) {
      await rollback();
      transactionStarted = false;
    }
  };

  try {
    const branchId = Number(req.params.branch_id || req.body.branch_id);
    const {
      shift_type_id: shiftTypeId,
      date,
      assignments,
      apply_to_dates: applyToDates,
      replace_existing: replaceExisting,
      remove,
    } = req.body;

    if (!branchId || !shiftTypeId || !date) {
      return res.status(400).json({
        error: "Branch, shift type, and primary date are required",
      });
    }

    await ensureBranchAccess(branchId, req.user.id);
    await ensureRosterEntityAccess("roster_shift_types", shiftTypeId, req.user.id);

    const dates =
      Array.isArray(applyToDates) && applyToDates.length > 0
        ? applyToDates
        : [date];

    if (!remove && (!Array.isArray(assignments) || assignments.length === 0)) {
      return res.status(400).json({
        error: "At least one assignment is required when not removing shifts",
      });
    }

    await beginTransaction();
    transactionStarted = true;

    const uniqueStaffIds = new Set();
    const uniquePositionIds = new Set();
    const uniqueBreakIds = new Set();

    if (!remove) {
      assignments.forEach((assignment) => {
        if (assignment.staff_id) uniqueStaffIds.add(assignment.staff_id);
        if (assignment.position_id) uniquePositionIds.add(assignment.position_id);
        if (Array.isArray(assignment.break_type_ids)) {
          assignment.break_type_ids.forEach((id) => uniqueBreakIds.add(id));
        }
      });

      if (uniqueStaffIds.size > 0) {
        const placeholders = Array.from(uniqueStaffIds)
          .map(() => "?")
          .join(",");
        const staffRows = await query(
          `
          SELECT id FROM staff 
          WHERE branch_id = ? AND id IN (${placeholders})
        `,
          [branchId, ...Array.from(uniqueStaffIds)]
        );
        if (staffRows.length !== uniqueStaffIds.size) {
          await rollbackIfNeeded();
          return res
            .status(400)
            .json({ error: "One or more staff members do not belong to this branch" });
        }
      }

      if (uniquePositionIds.size > 0) {
        const placeholders = Array.from(uniquePositionIds)
          .map(() => "?")
          .join(",");
        const positionRows = await query(
          `
          SELECT id FROM roster_positions
          WHERE branch_id = ? AND id IN (${placeholders})
        `,
          [branchId, ...Array.from(uniquePositionIds)]
        );
        if (positionRows.length !== uniquePositionIds.size) {
          await rollbackIfNeeded();
          return res
            .status(400)
            .json({ error: "One or more positions do not belong to this branch" });
        }
      }

      if (uniqueBreakIds.size > 0) {
        const placeholders = Array.from(uniqueBreakIds)
          .map(() => "?")
          .join(",");
        const breakRows = await query(
          `
          SELECT id FROM roster_break_types
          WHERE branch_id = ? AND id IN (${placeholders})
        `,
          [branchId, ...Array.from(uniqueBreakIds)]
        );
        if (breakRows.length !== uniqueBreakIds.size) {
          await rollbackIfNeeded();
          return res
            .status(400)
            .json({ error: "One or more breaks do not belong to this branch" });
        }
      }
    }

    if (remove) {
      const placeholders = dates.map(() => "?").join(", ");
      await query(
        `
        DELETE FROM roster_assignments
        WHERE branch_id = ?
          AND shift_type_id = ?
          AND shift_date IN (${placeholders})
      `,
        [branchId, shiftTypeId, ...dates]
      );
    }

    for (const targetDate of dates) {
      if (replaceExisting || remove) {
        await query(
          `
          DELETE FROM roster_assignments
          WHERE branch_id = ? AND shift_type_id = ? AND shift_date = ?
        `,
          [branchId, shiftTypeId, targetDate]
        );
      }

      if (remove) {
        continue;
      }

      for (const assignment of assignments) {
        if (!assignment.staff_id) {
          await rollbackIfNeeded();
          return res
            .status(400)
            .json({ error: "Each assignment requires a staff member" });
        }

        if (
          assignment.start_minutes === undefined ||
          assignment.end_minutes === undefined ||
          Number(assignment.start_minutes) >= Number(assignment.end_minutes)
        ) {
          await rollbackIfNeeded();
          return res.status(400).json({
            error: "Each assignment requires a valid start and end time",
          });
        }

        await query(
          `
          INSERT INTO roster_assignments
            (branch_id, shift_date, staff_id, shift_type_id, position_id, start_minutes, end_minutes, break_type_ids, notes, created_by_type, created_by_id)
          VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        `,
          [
            branchId,
            targetDate,
            assignment.staff_id,
            shiftTypeId,
            assignment.position_id || null,
            assignment.start_minutes,
            assignment.end_minutes,
            assignment.break_type_ids
              ? JSON.stringify(assignment.break_type_ids)
              : JSON.stringify([]),
            assignment.notes || null,
            req.user.role,
            req.user.role === "staff" ? req.user.staff_id : req.user.id,
          ]
        );
      }
    }

    await commit();
    transactionStarted = false;

    const minDate = dates.reduce(
      (min, current) => (current < min ? current : min),
      dates[0]
    );
    const maxDate = dates.reduce(
      (max, current) => (current > max ? current : max),
      dates[0]
    );

    const updatedRows = await query(
      `
      SELECT 
        ra.*,
        st.full_name,
        st.staff_code,
        st.phone,
        st.position,
        rst.name AS shift_type_name,
        rst.start_minutes AS default_start_minutes,
        rst.end_minutes AS default_end_minutes,
        rp.name AS position_name
      FROM roster_assignments ra
      JOIN staff st ON ra.staff_id = st.id
      JOIN roster_shift_types rst ON ra.shift_type_id = rst.id
      LEFT JOIN roster_positions rp ON ra.position_id = rp.id
      WHERE ra.branch_id = ? AND ra.shift_type_id = ?
        AND ra.shift_date BETWEEN ? AND ?
      ORDER BY ra.shift_date ASC, rst.start_minutes ASC
    `,
      [branchId, shiftTypeId, minDate, maxDate]
    );

    const breakTypes = await query(
      "SELECT * FROM roster_break_types WHERE branch_id = ?",
      [branchId]
    );
    const breakMap = getBreakTypeMap(breakTypes);

    const updatedAssignments = updatedRows.map((row) => {
      let breakIds = [];
      if (row.break_type_ids) {
        try {
          const parsed =
            typeof row.break_type_ids === "string"
              ? JSON.parse(row.break_type_ids)
              : row.break_type_ids;
          if (Array.isArray(parsed)) {
            breakIds = parsed;
          }
        } catch (parseError) {
          console.warn("Failed to parse break IDs for assignment", row.id);
        }
      }

      const breaks = breakIds.map((id) => breakMap.get(id)).filter(Boolean);

      return {
        id: row.id,
        branch_id: row.branch_id,
        shift_date: row.shift_date,
        staff: {
          id: row.staff_id,
          full_name: row.full_name,
          staff_code: row.staff_code,
          phone: row.phone,
          position: row.position,
        },
        shift_type: {
          id: row.shift_type_id,
          name: row.shift_type_name,
          startMinutes: row.start_minutes ?? row.default_start_minutes,
          endMinutes: row.end_minutes ?? row.default_end_minutes,
        },
        position: row.position_id
          ? {
              id: row.position_id,
              name: row.position_name,
            }
          : null,
        start_minutes: row.start_minutes,
        end_minutes: row.end_minutes,
        breaks,
        notes: row.notes,
      };
    });

    res.status(201).json({
      message: remove ? "Shift assignments removed" : "Shift assignments saved",
      assignments: updatedAssignments,
    });
  } catch (error) {
    await rollbackIfNeeded();
    console.error("Upsert roster assignments error:", error);
    res.status(error.status || 500).json({
      error: error.message || "Failed to save roster assignments",
    });
  }
}

module.exports = upsertAssignments;

