const {
  query,
  ensureBranchAccess,
  normalizeRole,
} = require("./shared");
const encryptionService = require("../../../services/encryptionService");

async function createStaff(req, res) {
  try {
    const {
      branch_id,
      staff_code,
      full_name,
      email,
      phone,
      position,
      hourly_rate,
      profile_image_path,
    } = req.body;

    if (!branch_id || !staff_code || !full_name) {
      return res.status(400).json({
        error: "Branch ID, staff code, and full name are required",
      });
    }

    // Verify branch belongs to this admin
    const branches = await query(
      `
      SELECT b.id 
      FROM branches b
      JOIN shops s ON b.shop_id = s.id
      WHERE b.id = ? AND s.admin_id = ?
    `,
      [branch_id, req.user.id],
    );

    if (branches.length === 0) {
      return res
        .status(403)
        .json({ error: "You do not have access to this branch" });
    }

    // Check if staff code already exists
    const existingStaff = await query(
      "SELECT id FROM staff WHERE staff_code = ?",
      [staff_code],
    );

    if (existingStaff.length > 0) {
      return res.status(400).json({ error: "Staff code already exists" });
    }

    // Encrypt sensitive data
    const encryptedFullName = encryptionService.encrypt(full_name);
    const encryptedEmail = email ? encryptionService.encrypt(email) : null;
    const encryptedPhone = phone ? encryptionService.encrypt(phone) : null;

    // Create staff
    const role = normalizeRole(position);

    const result = await query(
      `INSERT INTO staff 
       (branch_id, staff_code, full_name, email, phone, position, face_embeddings, hourly_rate, profile_image_path) 
       VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        branch_id,
        staff_code,
        encryptedFullName,
        encryptedEmail,
        encryptedPhone,
        role,
        null,
        hourly_rate !== undefined ? hourly_rate : 0,
        profile_image_path || null,
      ],
    );

    res.status(201).json({
      message: "Staff created successfully",
      staff: {
        id: result.insertId,
        branch_id,
        staff_code,
        full_name, // Return unencrypted for client
        email,
        phone,
        position: role,
        hourly_rate: hourly_rate !== undefined ? hourly_rate : 0,
      },
    });
  } catch (error) {
    console.error("Create staff error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
}

module.exports = createStaff;

