const {
  query,
  ensureBranchAccess,
  normalizeRole,
} = require("./shared");
const encryptionService = require("../../../services/encryptionService");
async function getStaffByBranch(req, res) {
  try {
    const { branch_id } = req.params;

    // Verify branch belongs to this admin
    const branch = await ensureBranchAccess(branch_id, req.user.id);

    // Get staff list
    const staff = await query(
      `
      SELECT 
        id,
        staff_code,
        full_name,
        email,
        phone,
        position,
        hourly_rate,
        profile_image_path,
        face_embeddings,
        is_active,
        created_at
      FROM staff
      WHERE branch_id = ?
      ORDER BY created_at DESC
    `,
      [branch_id],
    );

    const staffWithDecryptedData = staff.map((member) => {
      member.position = normalizeRole(member.position);
      
      // Decrypt sensitive fields
      try {
        member.full_name = member.full_name 
          ? encryptionService.decrypt(member.full_name) 
          : null;
        member.email = member.email 
          ? encryptionService.decrypt(member.email) 
          : null;
        member.phone = member.phone 
          ? encryptionService.decrypt(member.phone) 
          : null;
      } catch (error) {
        console.warn(`Failed to decrypt data for staff ${member.id}:`, error.message);
      }
      
      // Handle face embeddings
      let embeddingPayload = member.face_embeddings;
      if (embeddingPayload) {
        try {
          // Try to decrypt if it's encrypted
          if (typeof embeddingPayload === 'string') {
            const decrypted = encryptionService.decrypt(embeddingPayload);
            embeddingPayload = JSON.parse(decrypted);
          }
        } catch (e) {
          // If decryption fails, try parsing as regular JSON
          try {
            if (typeof embeddingPayload === "string") {
              embeddingPayload = JSON.parse(embeddingPayload);
            }
          } catch (e2) {
            console.warn(
              "Failed to parse face embeddings for staff",
              member.id,
              e2,
            );
            embeddingPayload = null;
          }
        }
      }

      if (embeddingPayload && typeof embeddingPayload === "object") {
        member.face_embeddings = embeddingPayload.image || null;
      } else {
        member.face_embeddings = null;
      }

      return member;
    });

    res.json({
      branch: branch,
      staff: staffWithDecryptedData,
    });
  } catch (error) {
    console.error("Get staff error:", error);
    res.status(error.status || 500).json({ error: error.message || "Internal server error" });
  }
}

module.exports = getStaffByBranch;

