const {
  query,
  normalizeRole,
  ensureStaffAccess,
} = require("./shared");
const encryptionService = require("../../../services/encryptionService");

async function getStaffById(req, res) {
  try {
    const { id } = req.params;

    await ensureStaffAccess(id, req.user.id);

    const rows = await query(
      `
      SELECT 
        st.*,
        b.name AS branch_name,
        s.name AS shop_name
      FROM staff st
      JOIN branches b ON st.branch_id = b.id
      JOIN shops s ON b.shop_id = s.id
      WHERE st.id = ?
    `,
      [id]
    );

    if (!rows || rows.length === 0) {
      return res.status(404).json({ error: "Staff not found" });
    }

    const staffMember = rows[0];
    staffMember.position = normalizeRole(staffMember.position);

    // Decrypt sensitive fields
    try {
      staffMember.full_name = staffMember.full_name 
        ? encryptionService.decrypt(staffMember.full_name) 
        : null;
      staffMember.email = staffMember.email 
        ? encryptionService.decrypt(staffMember.email) 
        : null;
      staffMember.phone = staffMember.phone 
        ? encryptionService.decrypt(staffMember.phone) 
        : null;
    } catch (error) {
      console.warn(`Failed to decrypt data for staff ${staffMember.id}:`, error.message);
    }

    // Handle face embeddings
    let embeddingPayload = staffMember.face_embeddings;
    if (embeddingPayload) {
      try {
        // Try to decrypt if it's encrypted
        if (typeof embeddingPayload === 'string') {
          const decrypted = encryptionService.decrypt(embeddingPayload);
          embeddingPayload = JSON.parse(decrypted);
        }
      } catch (e) {
        // If decryption fails, try parsing as regular JSON
        try {
          if (typeof embeddingPayload === "string") {
            embeddingPayload = JSON.parse(embeddingPayload);
          }
        } catch (e2) {
          console.warn(
            "Failed to parse face embeddings for staff",
            staffMember.id,
            e2,
          );
          embeddingPayload = null;
        }
      }
    }

    if (embeddingPayload && typeof embeddingPayload === "object") {
      staffMember.face_embeddings = embeddingPayload.image || null;
    } else {
      staffMember.face_embeddings = null;
    }

    res.json({
      staff: {
        id: staffMember.id,
        branch_id: staffMember.branch_id,
        staff_code: staffMember.staff_code,
        full_name: staffMember.full_name,
        email: staffMember.email,
        phone: staffMember.phone,
        position: staffMember.position,
        hourly_rate: staffMember.hourly_rate,
        profile_image_path: staffMember.profile_image_path,
        face_embeddings: staffMember.face_embeddings,
        is_active: staffMember.is_active === null || staffMember.is_active === undefined
          ? null
          : !!staffMember.is_active,
        created_at: staffMember.created_at,
        branch_name: staffMember.branch_name,
        shop_name: staffMember.shop_name,
      },
    });
  } catch (error) {
    console.error("Get staff by id error:", error);
    res.status(error.status || 500).json({ error: error.message || "Internal server error" });
  }
}

module.exports = getStaffById;

