const util = require("util");
const conn = require("../../../services/db");

const query = util.promisify(conn.query).bind(conn);

const normalizeRole = (role) => {
  if (typeof role !== "string") {
    return "staff";
  }
  const trimmed = role.trim();
  return trimmed.length > 0 ? trimmed : "staff";
};

const parseFaceEmbeddings = (value, staffId) => {
  if (!value) {
    return null;
  }

  let payload = value;
  try {
    if (typeof payload === "string") {
      payload = JSON.parse(payload);
    }
  } catch (error) {
    console.warn("Failed to parse face embeddings for staff", staffId, error);
    return null;
  }

  if (payload && typeof payload === "object") {
    return payload.image || null;
  }

  return null;
};

const ensureBranchAccess = async (branchId, adminId) => {
  const rows = await query(
    `
    SELECT b.id, b.name, s.name AS shop_name
    FROM branches b
    JOIN shops s ON b.shop_id = s.id
    WHERE b.id = ? AND s.admin_id = ?
  `,
    [branchId, adminId]
  );

  if (!rows || rows.length === 0) {
    const error = new Error("You do not have access to this branch");
    error.status = 403;
    throw error;
  }

  return rows[0];
};

const ensureStaffAccess = async (staffId, adminId) => {
  const rows = await query(
    `
    SELECT st.id, st.branch_id
    FROM staff st
    JOIN branches b ON st.branch_id = b.id
    JOIN shops s ON b.shop_id = s.id
    WHERE st.id = ? AND s.admin_id = ?
  `,
    [staffId, adminId]
  );

  if (!rows || rows.length === 0) {
    const error = new Error("You do not have access to this staff member");
    error.status = 403;
    throw error;
  }

  return rows[0];
};

const formatStaffRecord = (record) => ({
  id: record.id,
  branch_id: record.branch_id,
  staff_code: record.staff_code,
  full_name: record.full_name,
  email: record.email,
  phone: record.phone,
  position: normalizeRole(record.position),
  hourly_rate: record.hourly_rate,
  profile_image_path: record.profile_image_path,
  face_embeddings: parseFaceEmbeddings(record.face_embeddings, record.id),
  is_active:
    record.is_active === null || record.is_active === undefined
      ? null
      : !!record.is_active,
  created_at: record.created_at,
});

module.exports = {
  query,
  normalizeRole,
  parseFaceEmbeddings,
  ensureBranchAccess,
  ensureStaffAccess,
  formatStaffRecord,
};

