const crypto = require("crypto");
const AppError = require("../../../utils/appError");
const { sendPasswordResetEmail } = require("../../../utils/emailService");
const getEnvConfig = require("../../../utils/Config");
const { conn } = require("./shared");

async function forgotPassword(req, res, next) {
  const { email } = req?.body;

  try {
    if (!email) {
      return res.status(400).json({
        status: "error",
        message: "Email is required",
      });
    }

    const trimmedEmail = email?.trim()?.toLowerCase();
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(trimmedEmail)) {
      return res.status(400).json({
        status: "error",
        message: "Please enter a valid email address",
      });
    }

    const query =
      "SELECT id, email, first_name, last_name, full_name, password FROM `users` WHERE `is_deleted` IS NULL AND email = ?";

    conn.query(query, [trimmedEmail], async function (err, result) {
      if (err) {
        return next(
          new AppError(err?.message || "Something went wrong, Please try again", 500),
        );
      }

      // Always return success message for security
      if (result?.length > 0) {
        const user = result[0];

        if (user.password) {
          const resetToken = crypto.randomBytes(32).toString("hex");
          const resetTokenExpiry = new Date(Date.now() + 60 * 60 * 1000); // 1 hour

          const updateQuery =
            "UPDATE `users` SET password_reset_token = ?, password_reset_token_expiry = ? WHERE id = ?";

          conn.query(
            updateQuery,
            [resetToken, resetTokenExpiry, user.id],
            async function (updateErr) {
              if (updateErr) {
                console.error("Error updating password reset token:", updateErr);
                return res.status(200).json({
                  status: "success",
                  message:
                    "If an account with that email exists, a password reset link has been sent.",
                });
              }

              const baseUrl =
                getEnvConfig("FRONTEND_URL") ||
                getEnvConfig("APP_URL") ||
                "http://localhost:3000";
              const emailResult = await sendPasswordResetEmail(
                trimmedEmail,
                user.full_name ||
                  `${user.first_name} ${user.last_name}`.trim() ||
                  "User",
                resetToken,
                baseUrl,
              );

              if (!emailResult.success) {
                console.error(
                  "Failed to send password reset email:",
                  emailResult.error,
                );
              }

              return res.status(200).json({
                status: "success",
                message:
                  "If an account with that email exists, a password reset link has been sent.",
              });
            },
          );
        } else {
          return res.status(200).json({
            status: "success",
            message:
              "If an account with that email exists, a password reset link has been sent.",
          });
        }
      } else {
        return res.status(200).json({
          status: "success",
          message:
            "If an account with that email exists, a password reset link has been sent.",
        });
      }
    });
  } catch (e) {
    console.log("Exception Error : Forgot Password ", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
}

module.exports = forgotPassword;


