const { dbCompat: db } = require("./shared");

// Get current user profile
async function getProfile(req, res) {
  try {
    if (req.user.role === "staff") {
      const [staffRows] = await db.query(
        `
        SELECT 
          st.id,
          st.full_name,
          st.email,
          st.phone,
          st.position,
          st.branch_id,
          b.name AS branch_name,
          b.shop_id,
          s.name AS shop_name
        FROM staff st
        JOIN branches b ON st.branch_id = b.id
        JOIN shops s ON b.shop_id = s.id
        WHERE st.id = ?
      `,
        [req.user.staff_id || req.user.id],
      );

      if (staffRows.length === 0) {
        return res.status(404).json({ error: "Staff profile not found" });
      }

      const staff = staffRows[0];
      return res.json({
        user: {
          id: staff.id,
          email: staff.email || `${staff.phone || "staff"}@staff.faceml.local`,
          full_name: staff.full_name,
          role: "staff",
          shops: null,
        },
        staff: {
          id: staff.id,
          full_name: staff.full_name,
          phone: staff.phone,
          position: staff.position,
          branch_id: staff.branch_id,
          branch_name: staff.branch_name,
          shop_id: staff.shop_id,
          shop_name: staff.shop_name,
        },
        hasSubscription: false, // Staff don't have subscriptions
      });
    }

    const [users] = await db.query(
      "SELECT id, email, full_name, role, created_at FROM users WHERE id = ?",
      [req.user.id],
    );

    if (users.length === 0) {
      return res.status(404).json({ error: "User not found" });
    }

    let shops = [];
    if (users[0].role === "shop_admin") {
      [shops] = await db.query(
        "SELECT id, name, address, contact_phone FROM shops WHERE admin_id = ? AND is_active = true",
        [req.user.id],
      );
      console.log(
        `Profile: Found ${shops.length} shops for user ${req.user.id}:`,
        shops,
      );
    }

    // Check for active subscription (best-effort)
    let subscriptionData = null;
    let hasSubscription = false;
    try {
      const [subscriptions] = await db.query(
        `SELECT 
          us.id as subscription_id,
          us.plan_type,
          us.amount,
          us.start_date,
          us.end_date,
          us.status as subscription_status,
          us.created_at as subscription_created_at,
          sp.id as plan_id,
          sp.name as plan_name,
          sp.price as plan_price,
          sp.duration_type,
          sp.max_locations,
          sp.max_staff,
          sp.description as plan_description
        FROM user_subscriptions us
        LEFT JOIN subscription_plans sp ON us.subscription_plan_id = sp.id
        WHERE us.user_id = ? 
          AND us.status = 'active' 
          AND us.end_date > NOW()
        ORDER BY us.created_at DESC
        LIMIT 1`,
        [req.user.id],
      );

      if (subscriptions && subscriptions.length > 0) {
        hasSubscription = true;
        subscriptionData = subscriptions[0];
      }
    } catch (subErr) {
      console.log("Error fetching subscription:", subErr);
      // continue without subscription data
    }

    res.json({
      user: {
        ...users[0],
        shops: shops,
      },
      hasSubscription: hasSubscription,
      subscription: subscriptionData,
    });
  } catch (error) {
    console.error("Get profile error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
}

module.exports = getProfile;


