const jwt = require("jsonwebtoken");
const { dbCompat: db } = require("./shared");
const { generateToken, generateRefreshToken } = require("./token");

async function refreshToken(req, res) {
  try {
    const { refreshToken } = req.body;

    if (!refreshToken) {
      return res.status(400).json({ error: "Refresh token is required" });
    }

    let decoded;
    try {
      decoded = jwt.verify(
        refreshToken,
        process.env.JWT_REFRESH_SECRET || process.env.JWT_SECRET + "_refresh",
      );
    } catch (error) {
      console.error("Token refresh error:", error);
      return res.status(401).json({ error: "Invalid or expired refresh token" });
    }

    let userData;
    if (decoded.role === "staff") {
      const [staffRows] = await db.query(
        `
        SELECT 
          st.*,
          b.shop_id,
          s.admin_id
        FROM staff st
        JOIN branches b ON st.branch_id = b.id
        JOIN shops s ON b.shop_id = s.id
        WHERE st.id = ? AND st.is_active = true
        `,
        [decoded.id],
      );

      if (staffRows.length === 0) {
        return res
          .status(404)
          .json({ error: "Staff member not found or inactive" });
      }

      const staff = staffRows[0];
      userData = {
        id: staff.id,
        email:
          staff.email || `${staff.staff_code || staff.phone}@staff.faceml.local`,
        role: "staff",
        full_name: staff.full_name,
        staff_id: staff.id,
        branch_id: staff.branch_id,
        shop_id: staff.shop_id,
        admin_id: staff.admin_id,
        phone: staff.phone,
      };
    } else {
      const [users] = await db.query(
        "SELECT * FROM users WHERE id = ? AND is_active = true",
        [decoded.id],
      );

      if (users.length === 0) {
        return res
          .status(404)
          .json({ error: "User not found or inactive" });
      }

      userData = users[0];
    }

    const newToken = generateToken(userData);
    const newRefreshToken = generateRefreshToken(userData);

    res.json({
      message: "Token refreshed successfully",
      token: newToken,
      refreshToken: newRefreshToken,
    });
  } catch (error) {
    console.error("Refresh token error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
}

module.exports = refreshToken;


