const crypto = require("crypto");
const { Hash } = require("../../../utils/encryption");
const DbHelper = require("../../../helpers/DbHelper");
const AppError = require("../../../utils/appError");
const { sendVerificationEmail } = require("../../../utils/emailService");
const getEnvConfig = require("../../../utils/Config");
const { conn } = require("./shared");

async function register(req, res, next) {
  const {
    firstName,
    lastName,
    email,
    phone,
    shopName,
    location,
    password,
    confirmPassword,
  } = req?.body;

  try {
    const trimmedFirstName = firstName?.trim();
    const trimmedLastName = lastName?.trim();
    const trimmedEmail = email?.trim()?.toLowerCase();
    const trimmedPhone = phone?.trim();
    const trimmedShopName = shopName?.trim();
    const trimmedLocation = location?.trim();
    const trimmedPassword = password?.trim();
    const trimmedConfirmPassword = confirmPassword?.trim();

    if (!trimmedFirstName) return next(new AppError("First name is required", 400));
    if (!trimmedLastName) return next(new AppError("Last name is required", 400));
    if (!trimmedEmail) return next(new AppError("Email is required", 400));
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(trimmedEmail))
      return next(new AppError("Please enter a valid email address", 400));
    if (!trimmedShopName) return next(new AppError("Business is required", 400));
    if (!trimmedPhone) return next(new AppError("Phone is required", 400));
    if (!trimmedLocation) return next(new AppError("Location is required", 400));
    if (!trimmedPassword) return next(new AppError("Password is required", 400));
    if (!trimmedConfirmPassword)
      return next(new AppError("Confirm password is required", 400));
    if (trimmedPassword.length < 6)
      return next(new AppError("Password must be at least 6 characters long", 400));
    if (trimmedPassword !== trimmedConfirmPassword)
      return next(new AppError("Password and confirm password do not match", 400));

    const checkEmailQuery =
      "SELECT id FROM `users` WHERE `is_deleted` IS NULL AND email = ?";
    conn.query(checkEmailQuery, [trimmedEmail], async function (err, result) {
      if (err) {
        return next(new AppError("Something went wrong, Please try again", 500));
      }

      if (result && result.length > 0) {
        return next(
          new AppError(
            "Email already exists. Please use a different email.",
            400,
          ),
        );
      }

      const full_name = `${trimmedFirstName} ${trimmedLastName}`.trim();
      const hashedPassword = await Hash(trimmedPassword);
      const verificationToken = crypto.randomBytes(32).toString("hex");
      const verificationTokenExpiry = new Date(Date.now() + 24 * 60 * 60 * 1000);

      const query =
        "INSERT INTO `users` (first_name, last_name, full_name, email, phone, shop_name, location, password, email_verified, verification_token, verification_token_expiry, role) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'shop_admin')";

      conn.query(
        query,
        [
          trimmedFirstName,
          trimmedLastName,
          full_name,
          trimmedEmail,
          trimmedPhone,
          trimmedShopName,
          trimmedLocation,
          hashedPassword,
          0,
          verificationToken,
          verificationTokenExpiry,
        ],
        async function (err2) {
          if (err2) {
            const isDuplicateEntry =
              err2.code === "ER_DUP_ENTRY" ||
              err2.code === 1062 ||
              err2.errno === 1062 ||
              (err2.message &&
                (err2.message.includes("Duplicate entry") ||
                  err2.message.includes("Duplicate") ||
                  err2.message.toLowerCase().includes("email")));

            if (isDuplicateEntry) {
              return next(
                new AppError(
                  "Email already exists. Please use a different email.",
                  400,
                ),
              );
            }
            return next(
              new AppError(err2?.message || "Something went wrong, Please try again", 500),
            );
          }

          const baseUrl =
            getEnvConfig("FRONTEND_URL") ||
            getEnvConfig("APP_URL") ||
            "http://localhost:3000";
          const emailResult = await sendVerificationEmail(
            trimmedEmail,
            full_name,
            verificationToken,
            baseUrl,
          );

          if (!emailResult.success) {
            console.error("Failed to send verification email:", emailResult.error);
          }

          return res.status(200).json({
            status: "success",
            message:
              "Registration successful! Please check your email to verify your account.",
          });
        },
      );
    });
  } catch (e) {
    console.log("Exception Error : User Registration ", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
}

module.exports = register;


