const { Hash } = require("../../../utils/encryption");
const AppError = require("../../../utils/appError");
const { conn } = require("./shared");

async function resetPassword(req, res, next) {
  let token = req?.query?.token || req?.body?.token;
  const { password, confirmPassword } = req?.body;

  if (token) {
    token = token.trim();
    try {
      token = decodeURIComponent(token);
    } catch (e) {
      console.log("Token decode warning:", e.message);
    }
  }

  try {
    if (!token) {
      return res.status(400).json({
        status: "error",
        message: "Reset token is required",
      });
    }

    if (!password || !confirmPassword) {
      return res.status(400).json({
        status: "error",
        message: "Password and confirm password are required",
      });
    }

    if (password !== confirmPassword) {
      return res.status(400).json({
        status: "error",
        message: "Password and confirm password do not match",
      });
    }

    if (password.length < 6) {
      return res.status(400).json({
        status: "error",
        message: "Password must be at least 6 characters long",
      });
    }

    const query =
      "SELECT id, email, password_reset_token_expiry FROM `users` WHERE `is_deleted` IS NULL AND password_reset_token = ?";

    conn.query(query, [token], async function (err, result) {
      if (err) {
        return next(
          new AppError(err?.message || "Something went wrong, Please try again", 500),
        );
      }

      if (!result || result.length === 0) {
        return res.status(400).json({
          status: "error",
          message:
            "Invalid or expired reset token. Please request a new password reset link.",
        });
      }

      const user = result[0];
      const now = new Date();
      const expiryDate = new Date(user.password_reset_token_expiry);
      if (expiryDate < now) {
        return res.status(400).json({
          status: "error",
          message:
            "Reset token has expired. Please request a new password reset link.",
        });
      }

      const hashedPassword = await Hash(password);

      const updateQuery =
        "UPDATE `users` SET password = ?, password_reset_token = NULL, password_reset_token_expiry = NULL WHERE id = ? AND password_reset_token = ?";

      conn.query(
        updateQuery,
        [hashedPassword, user.id, token],
        async function (updateErr, updateResult) {
          if (updateErr) {
            return next(
              new AppError(
                updateErr?.message || "Something went wrong, Please try again",
                500,
              ),
            );
          }

          if (updateResult && updateResult.affectedRows > 0) {
            return res.status(200).json({
              status: "success",
              message:
                "Your password has been reset successfully. You can now log in with your new password.",
            });
          }

          return res.status(400).json({
            status: "error",
            message:
              "Failed to reset password. Please try again or request a new reset link.",
          });
        },
      );
    });
  } catch (e) {
    console.log("Exception Error : Reset Password ", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
}

module.exports = resetPassword;


