const { dbCompat: db } = require("./shared");
const { generateToken, generateRefreshToken } = require("./token");

// Staff login via phone
async function staffLogin(req, res) {
  try {
    const { phone, staff_code: staffCode } = req.body;

    if (!phone) {
      return res.status(400).json({ error: "Phone number is required" });
    }

    let query = `
      SELECT 
        st.*,
        b.name AS branch_name,
        b.address AS branch_address,
        b.contact_phone AS branch_phone,
        b.is_active AS branch_active,
        s.id AS shop_id,
        s.name AS shop_name,
        s.admin_id,
        s.address AS shop_address,
        s.contact_phone AS shop_phone
      FROM staff st
      JOIN branches b ON st.branch_id = b.id
      JOIN shops s ON b.shop_id = s.id
      WHERE st.phone = ? AND st.is_active = true
    `;
    const params = [phone];

    if (staffCode) {
      query += " AND st.staff_code = ?";
      params.push(staffCode);
    }

    const [staffRows] = await db.query(query, params);

    if (staffRows.length === 0) {
      return res.status(401).json({ error: "Staff member not found" });
    }

    const staff = staffRows[0];

    const tokenPayload = {
      id: staff.id,
      email: staff.email || `${staff.staff_code || staff.phone}@staff.faceml.local`,
      role: "staff",
      full_name: staff.full_name,
      staff_id: staff.id,
      branch_id: staff.branch_id,
      shop_id: staff.shop_id,
      admin_id: staff.admin_id,
      phone: staff.phone,
    };

    const token = generateToken(tokenPayload);
    const refreshToken = generateRefreshToken(tokenPayload);

    const userPayload = {
      id: staff.id,
      email: staff.email || `${staff.staff_code || staff.phone}@staff.faceml.local`,
      full_name: staff.full_name,
      role: "staff",
      shops: null,
    };

    res.json({
      status: "success",
      message: "Staff login successful",
      data: {
        accessToken: token,
        refreshToken: refreshToken,
        user: userPayload,
        staff: {
          id: staff.id,
          full_name: staff.full_name,
          staff_code: staff.staff_code,
          phone: staff.phone,
          position: staff.position,
          hourly_rate: staff.hourly_rate,
          branch_id: staff.branch_id,
        },
        branch: {
          id: staff.branch_id,
          name: staff.branch_name,
          shop_id: staff.shop_id,
          address: staff.branch_address,
          contact_phone: staff.branch_phone,
          is_active: staff.branch_active,
        },
        shop: {
          id: staff.shop_id,
          name: staff.shop_name,
          address: staff.shop_address,
          contact_phone: staff.shop_phone,
        },
      },
    });
  } catch (error) {
    console.error("Staff login error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
}

module.exports = staffLogin;


