const {
  query,
  ensureBranchAccess,
  normalizeAvailabilityRecord,
} = require("./shared");

async function getBranchAvailability(req, res) {
  try {
    const branchId = Number(req.params.branch_id);
    if (Number.isNaN(branchId)) {
      return res.status(400).json({ error: "Invalid branch ID" });
    }

    const { start_date: startDate, end_date: endDate } = req.query;

    const branchContext = await ensureBranchAccess(branchId, req.user.id);

    let availabilityQuery = `
      SELECT 
        sa.*,
        st.full_name,
        st.staff_code,
        st.phone,
        st.position
      FROM staff_availability sa
      JOIN staff st ON sa.staff_id = st.id
      WHERE sa.branch_id = ?
    `;
    const params = [branchId];

    if (startDate) {
      availabilityQuery += " AND sa.availability_date >= ?";
      params.push(startDate);
    }

    if (endDate) {
      availabilityQuery += " AND sa.availability_date <= ?";
      params.push(endDate);
    }

    availabilityQuery += " ORDER BY sa.availability_date ASC, st.full_name ASC";

    const rows = await query(availabilityQuery, params);

    res.json({
      branch: {
        id: branchContext.id,
        name: branchContext.name,
        shop_id: branchContext.shop_id,
        shop_name: branchContext.shop_name,
      },
      availability: rows.map((record) => ({
        ...normalizeAvailabilityRecord(record),
        staff: {
          id: record.staff_id,
          full_name: record.full_name,
          staff_code: record.staff_code,
          phone: record.phone,
          position: record.position,
        },
      })),
    });
  } catch (error) {
    console.error("Get branch availability error:", error);
    res
      .status(error.status || 500)
      .json({ error: error.message || "Failed to fetch branch availability" });
  }
}

module.exports = getBranchAvailability;

