const {
  query,
  ensureStaffAccess,
  normalizeAvailabilityRecord,
} = require("./shared");

async function getStaffAvailability(req, res) {
  try {
    const staffId = Number(req.params.staff_id);
    if (Number.isNaN(staffId)) {
      return res.status(400).json({ error: "Invalid staff ID" });
    }

    const { start_date: startDate, end_date: endDate } = req.query;

    const staffContext = await ensureStaffAccess(req.user, staffId);

    let availabilityQuery = `
      SELECT *
      FROM staff_availability
      WHERE staff_id = ?
    `;
    const params = [staffId];

    if (startDate) {
      availabilityQuery += " AND availability_date >= ?";
      params.push(startDate);
    }

    if (endDate) {
      availabilityQuery += " AND availability_date <= ?";
      params.push(endDate);
    }

    availabilityQuery += " ORDER BY availability_date ASC";

    const availabilityRows = await query(availabilityQuery, params);

    res.json({
      staff: {
        id: staffContext.id,
        full_name: staffContext.full_name,
        staff_code: staffContext.staff_code,
        phone: staffContext.phone,
        position: staffContext.position,
        branch_id: staffContext.branch_id,
        branch_name: staffContext.branch_name,
        shop_id: staffContext.shop_id,
        shop_name: staffContext.shop_name,
      },
      availability: availabilityRows.map(normalizeAvailabilityRecord),
    });
  } catch (error) {
    console.error("Get staff availability error:", error);
    res
      .status(error.status || 500)
      .json({ error: error.message || "Failed to fetch availability" });
  }
}

module.exports = getStaffAvailability;

