const util = require("util");
const conn = require("../../../services/db");

const query = util.promisify(conn.query).bind(conn);
const beginTransaction = util.promisify(conn.beginTransaction).bind(conn);
const commit = util.promisify(conn.commit).bind(conn);
const rollback = util.promisify(conn.rollback).bind(conn);

const fetchStaffContext = async (staffId) => {
  const rows = await query(
    `
    SELECT 
      st.id,
      st.branch_id,
      st.full_name,
      st.staff_code,
      st.phone,
      st.position,
      b.name AS branch_name,
      b.shop_id,
      s.name AS shop_name,
      s.admin_id
    FROM staff st
    JOIN branches b ON st.branch_id = b.id
    JOIN shops s ON b.shop_id = s.id
    WHERE st.id = ?
  `,
    [staffId]
  );

  return rows && rows.length > 0 ? rows[0] : null;
};

const ensureStaffAccess = async (user, staffId) => {
  const context = await fetchStaffContext(staffId);

  if (!context) {
    const error = new Error("Staff member not found");
    error.status = 404;
    throw error;
  }

  if (user.role === "staff") {
    if (user.staff_id !== context.id) {
      const error = new Error("You cannot access another staff member");
      error.status = 403;
      throw error;
    }
  } else if (user.role === "shop_admin") {
    if (context.admin_id !== user.id) {
      const error = new Error("You do not have access to this staff member");
      error.status = 403;
      throw error;
    }
  } else {
    const error = new Error("Unauthorized role");
    error.status = 403;
    throw error;
  }

  return context;
};

const ensureBranchAccess = async (branchId, adminId) => {
  const rows = await query(
    `
    SELECT 
      b.id,
      b.name,
      b.shop_id,
      s.name AS shop_name,
      s.admin_id
    FROM branches b
    JOIN shops s ON b.shop_id = s.id
    WHERE b.id = ?
  `,
    [branchId]
  );

  if (!rows || rows.length === 0) {
    const error = new Error("Branch not found");
    error.status = 404;
    throw error;
  }

  if (rows[0].admin_id !== adminId) {
    const error = new Error("You do not have access to this branch");
    error.status = 403;
    throw error;
  }

  return rows[0];
};

const normalizeAvailabilityRecord = (record) => ({
  id: record.id,
  staff_id: record.staff_id,
  branch_id: record.branch_id,
  date: record.availability_date,
  is_available: Boolean(record.is_available),
  start_minutes: record.start_minutes,
  end_minutes: record.end_minutes,
  notes: record.notes,
  submitted_by: record.submitted_by,
  submitted_by_id: record.submitted_by_id,
  created_at: record.created_at,
  updated_at: record.updated_at,
});

module.exports = {
  query,
  beginTransaction,
  commit,
  rollback,
  ensureStaffAccess,
  ensureBranchAccess,
  normalizeAvailabilityRecord,
};

