const {
  query,
  beginTransaction,
  commit,
  rollback,
  ensureStaffAccess,
  normalizeAvailabilityRecord,
} = require("./shared");

async function upsertStaffAvailability(req, res) {
  let transactionStarted = false;

  const rollbackIfNeeded = async () => {
    if (transactionStarted) {
      await rollback();
      transactionStarted = false;
    }
  };

  try {
    const staffId = Number(req.params.staff_id);
    if (Number.isNaN(staffId)) {
      return res.status(400).json({ error: "Invalid staff ID" });
    }

    const { branch_id: branchIdInput, entries } = req.body;
    if (!Array.isArray(entries) || entries.length === 0) {
      return res.status(400).json({ error: "Entries array is required" });
    }

    const staffContext = await ensureStaffAccess(req.user, staffId);
    const branchId = branchIdInput || staffContext.branch_id;

    if (branchId !== staffContext.branch_id) {
      return res
        .status(400)
        .json({ error: "Branch mismatch for this staff member" });
    }

    const submitterRole = req.user.role === "staff" ? "staff" : "shop_admin";
    const submitterId =
      req.user.role === "staff" ? req.user.staff_id : req.user.id;

    await beginTransaction();
    transactionStarted = true;

    for (const entry of entries) {
      const {
        date,
        is_available: isAvailable,
        start_minutes: startMinutes,
        end_minutes: endMinutes,
        notes,
      } = entry;

      if (!date) {
        await rollbackIfNeeded();
        return res.status(400).json({ error: "Each entry requires a date" });
      }

      if (
        isAvailable !== false &&
        (startMinutes === undefined || endMinutes === undefined)
      ) {
        await rollbackIfNeeded();
        return res.status(400).json({
          error: "Start and end minutes are required for available days",
        });
      }

      if (
        isAvailable !== false &&
        Number(startMinutes) >= Number(endMinutes)
      ) {
        await rollbackIfNeeded();
        return res
          .status(400)
          .json({ error: "End time must be after start time" });
      }

      await query(
        `
        INSERT INTO staff_availability (
          staff_id,
          branch_id,
          availability_date,
          is_available,
          start_minutes,
          end_minutes,
          notes,
          submitted_by,
          submitted_by_id
        )
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE
          is_available = VALUES(is_available),
          start_minutes = VALUES(start_minutes),
          end_minutes = VALUES(end_minutes),
          notes = VALUES(notes),
          submitted_by = VALUES(submitted_by),
          submitted_by_id = VALUES(submitted_by_id),
          updated_at = CURRENT_TIMESTAMP
      `,
        [
          staffId,
          branchId,
          date,
          isAvailable === false ? false : true,
          isAvailable === false ? null : startMinutes,
          isAvailable === false ? null : endMinutes,
          notes || null,
          submitterRole,
          submitterId,
        ]
      );
    }

    await commit();
    transactionStarted = false;

    const dates = entries.map((entry) => entry.date).filter(Boolean);
    const minDate = dates.reduce(
      (min, current) => (current < min ? current : min),
      dates[0]
    );
    const maxDate = dates.reduce(
      (max, current) => (current > max ? current : max),
      dates[0]
    );

    let fetchQuery = `
      SELECT *
      FROM staff_availability
      WHERE staff_id = ?
    `;
    const fetchParams = [staffId];

    if (minDate && maxDate) {
      fetchQuery += " AND availability_date BETWEEN ? AND ?";
      fetchParams.push(minDate, maxDate);
    }

    fetchQuery += " ORDER BY availability_date ASC";

    const updatedEntries = await query(fetchQuery, fetchParams);

    res.status(201).json({
      message: "Availability saved successfully",
      availability: updatedEntries.map(normalizeAvailabilityRecord),
    });
  } catch (error) {
    await rollbackIfNeeded();
    console.error("Upsert availability error:", error);
    res
      .status(error.status || 500)
      .json({ error: error.message || "Failed to save availability" });
  }
}

module.exports = upsertStaffAvailability;

