const DbHelper = require("../../../helpers/DbHelper");
const AppError = require("../../../utils/appError");
const conn = require("../../../services/db");

async function checkActiveSubscription(req, res, next) {
  try {
    const { user_id } = req?.headers;

    if (!user_id) {
      return next(new AppError("User authentication required", 401));
    }

    // Check if user has an active or cancelled (but not expired) subscription
    const query = `
      SELECT 
        us.id,
        us.plan_type,
        us.amount,
        us.start_date,
        us.end_date,
        us.status,
        us.subscription_plan_id,
        us.stripe_subscription_id,
        sp.name as plan_name,
        sp.duration_type
      FROM user_subscriptions us
      LEFT JOIN subscription_plans sp ON us.subscription_plan_id = sp.id
      WHERE us.user_id = ? 
        AND (us.status = 'active' OR (us.status = 'cancelled' AND us.end_date > NOW()))
        AND us.end_date > NOW()
      ORDER BY us.created_at DESC
      LIMIT 1
    `;

    const result = await DbHelper.promisifyQuery(query, conn, next, [user_id]);

    if (result && result.length > 0) {
      // Convert amount from string to number (MySQL DECIMAL returns as string)
      const subscription = {
        ...result[0],
        amount: parseFloat(result[0].amount) || 0,
      };
      
      return res.status(200).json({
        status: "success",
        hasActiveSubscription: true,
        data: subscription,
      });
    }

    res.status(200).json({
      status: "success",
      hasActiveSubscription: false,
      data: null,
    });
  } catch (e) {
    console.log("Exception Error : Check Active Subscription ", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
}

module.exports = checkActiveSubscription;

