const AppError = require("../../../utils/appError");
const conn = require("../../../services/db");
const DbHelper = require("../../../helpers/DbHelper");
const stripe = require("stripe")(process.env.STRIPE_SECRET_KEY);

async function reactivateSubscription(req, res, next) {
  try {
    const { user_id } = req?.headers;

    if (!user_id) {
      return next(new AppError("User authentication required", 401));
    }

    // Find user's cancelled subscription with Stripe subscription ID
    const findSubscriptionQuery = `
      SELECT 
        us.id,
        us.stripe_subscription_id,
        us.status,
        us.end_date,
        us.subscription_plan_id,
        sp.stripe_price_id
      FROM user_subscriptions us
      LEFT JOIN subscription_plans sp ON us.subscription_plan_id = sp.id
      WHERE us.user_id = ? 
        AND us.status = 'cancelled'
        AND us.stripe_subscription_id IS NOT NULL
        AND us.end_date > NOW()
      ORDER BY us.created_at DESC
      LIMIT 1
    `;

    const subscriptionResult = await DbHelper.promisifyQuery(
      findSubscriptionQuery,
      conn,
      next,
      [user_id]
    );

    if (!subscriptionResult || subscriptionResult.length === 0) {
      return res.status(404).json({
        status: "error",
        message: "No cancelled subscription found that can be reactivated",
      });
    }

    const subscription = subscriptionResult[0];

    try {
      // Reactivate subscription in Stripe
      const reactivatedSubscription = await stripe.subscriptions.update(subscription.stripe_subscription_id, {
        cancel_at_period_end: false,
      });

      // Update database
      const startDate = new Date(reactivatedSubscription.current_period_start * 1000).toISOString().slice(0, 19).replace('T', ' ');
      const endDate = new Date(reactivatedSubscription.current_period_end * 1000).toISOString().slice(0, 19).replace('T', ' ');

      const updateQuery = `
        UPDATE user_subscriptions 
        SET status = 'active',
            start_date = ?,
            end_date = ?,
            updated_at = NOW()
        WHERE id = ?
      `;

      await DbHelper.promisifyQuery(
        updateQuery,
        conn,
        next,
        [startDate, endDate, subscription.id]
      );

      return res.status(200).json({
        status: "success",
        message: "Your subscription has been reactivated successfully. Auto-renewal has been enabled.",
        data: {
          subscriptionId: subscription.id,
          status: 'active',
          startDate: startDate,
          endDate: endDate,
        },
      });
    } catch (stripeError) {
      console.error("Stripe reactivation error:", stripeError);
      
      // If subscription doesn't exist in Stripe, return error
      if (stripeError.code === 'resource_missing') {
        return next(new AppError("Subscription not found in Stripe. Please contact support.", 404));
      }

      return next(new AppError(stripeError.message || "Failed to reactivate subscription", 500));
    }
  } catch (e) {
    console.log("Exception Error: Reactivate Subscription", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
}

module.exports = reactivateSubscription;

