const AppError = require("../utils/appError");
const conn = require("../services/db").default;
const util = require("util");

exports.getAllMenu = (req, res, next) => {
  conn.query(
    "SELECT `title`,`slug` FROM `category` WHERE `status` = ? AND `show_in_menu` = ?",
    ["active", 1],
    function (err, data, fields) {
      if (err) return next(new AppError(err));
      res.status(200).json({
        status: "success",
        length: data?.length,
        data: data,
      });
    }
  );
};

exports.subscribeEmail = (req, res, next) => {
  if (!req.body) return next(new AppError("No form data found", 404));
  const { email } = req?.body;
  if (!email)
    return next(new AppError("Missing Email address field value", 500));
  conn.query(
    "SELECT * FROM `subscriptions` WHERE `email` = ?",
    [email],
    function (err, data, fields) {
      if (err) return next(new AppError(err, 500));
      if (data?.length > 0)
        return next(
          new AppError(
            "Apologies, the email address you've provided is already subscribed. Kindly consider using an alternative email address.",
            500
          )
        );
      conn.query(
        "INSERT INTO `subscriptions` (`email`) VALUES (?)",
        [email],
        function (err, data, fields) {
          if (err) return next(new AppError(err, 500));
          res.status(201).json({
            status: "success",
            message:
              "Congratulations, You have successfully subscribed with our notification channel to get latest notifications and updates from our website.",
          });
        }
      );
    }
  );
};

exports.notificationToken = (req, res, next) => {
  if (!req.body) return next(new AppError("No form data found", 404));
  const { token } = req?.body;
  if (!token)
    return next(new AppError("Token field value cannot be empty", 500));
  conn.query(
    "SELECT * FROM `firebase_keys` WHERE `key` = ?",
    [token],
    function (err, data, fields) {
      if (err) return next(new AppError(err, 500));
      if (data?.length > 0) return next(new AppError("declined", 500));
      conn.query(
        "INSERT INTO `firebase_keys` (`key`) VALUES (?)",
        [token],
        function (err, data, fields) {
          if (err) return next(new AppError(err, 500));
          res.status(201).json({
            status: "success",
            message: "Notification token stored",
          });
        }
      );
    }
  );
};

exports.getBlog = (req, res, next) => {
  if (!req.params) return next(new AppError("No form data found", 404));
  const { slug } = req?.params;
  if (!slug) return next(new AppError("Missing blog name", 500));
  conn.query(
    "SELECT p.* , GROUP_CONCAT(p2.category_id) as categories, GROUP_CONCAT(cat.title) as categoryName FROM `posts` as p LEFT JOIN `post_categories` as p2 ON p2.post_id = p.id LEFT JOIN `category` as cat ON cat.`id` = p2.`category_id` WHERE p.`slug` = ? AND p.`status` = ? GROUP BY p.id",
    [slug, "active"],
    function (err, data, fields) {
      if (err) return next(new AppError(err, 500));
      if (data.length < 1) return next(new AppError("Blog not found", 404));
      const categories = data.length > 0 ? data[0]?.categories : "";
      if (categories) {
        conn.query(
          "SELECT * FROM `category` WHERE id IN (" +
            categories +
            ") AND status = ?",
          ["active"],
          function (err, catResults, fields) {
            console.log("results", categories, catResults);
            if (err) return next(new AppError(err, 500));
            res.status(201).json({
              status: "success",
              message: "Blog Information",
              data: {
                ...data[0],
                relatedCategories: catResults,
              },
            });
          }
        );
      } else {
        res.status(201).json({
          status: "success",
          message: "Blog Information",
          data,
        });
      }
    }
  );
};

exports.getCategory = (req, res, next) => {
  if (!req.params) return next(new AppError("No req data found", 404));
  const { slug } = req?.params;
  if (!slug) return next(new AppError("Missing category name", 500));
  conn.query(
    "SELECT * FROM `category` WHERE `slug` = ? AND `status` = ?",
    [slug, "active"],
    function (err, data, fields) {
      if (err) return next(new AppError(err, 500));
      if (data.length < 1) return next(new AppError("Category not found", 404));
      const categoryId = data.length > 0 ? data[0]?.id : "";
      if (categoryId) {
        conn.query(
          "SELECT p.* FROM `posts` as p JOIN `post_categories` as pc ON pc.post_id = p.id WHERE pc.category_id = ? AND p.status = ? ORDER BY p.id",
          [categoryId, "active"],
          function (err, postResult, fields) {
            if (err) return next(new AppError(err, 500));
            res.status(201).json({
              status: "success",
              message: "Category Information",
              data: {
                ...data[0],
                relatedBlogs: postResult,
              },
            });
          }
        );
      } else {
        res.status(201).json({
          status: "success",
          message: "Category Information",
          data,
        });
      }
    }
  );
};

exports.getCMS = (req, res, next) => {
  if (!req.params) return next(new AppError("No req data found", 404));
  const { slug } = req.params;
  if (!slug) return next(new AppError("Missing page name", 500));
  conn.query(
    "SELECT title,description,meta_title,meta_keywords,meta_description FROM `pages` WHERE `slug` = ? AND `status` = ?",
    [slug, "active"],
    function (err, data, fields) {
      if (err) return next(new AppError(err, 500));
      if (data.length < 1) return next(new AppError("Page not found", 404));
      res.status(201).json({
        status: "success",
        message: "Page Information",
        data: data[0],
      });
    }
  );
};

exports.storeContactRequest = (req, res, next) => {
  if (!req.body) return next(new AppError("No req data found", 404));
  const { name, subject, email, phone, message } = req?.body;
  if (!name || !subject || !email || !phone || !message) {
    return next(
      new AppError(
        "Name,Subject,Email,Phone and Message field can not be empty",
        500
      )
    );
  }

  conn.query(
    "INSERT INTO `contact_manager` (`name`,`subject`,`email`,`phone`,`message`) VALUES (?,?,?,?,?)",
    [name, subject, email, phone, message],
    function (err, data, fields) {
      if (err) return next(new AppError(err, 500));
      if (data.length < 1) return next(new AppError("Page not found", 404));
      res.status(201).json({
        status: "success",
        message: "Contact Request has been sent successfully",
      });
    }
  );
};

exports.getHomeConfiguration = async (req, res, next) => {
  let posts = "";
  let trendingPosts = "";
  let categories = "";
  let featuredCategory = "";
  await promiseQuery(
    "SELECT * FROM `category` WHERE status = 'active' AND is_home_category = 1 LIMIT 4",
    next
  )
    .then((value) => {
      categories = value;
    })
    .catch((err) => {
      return next(new AppError(err, 500));
    });

  await promiseQuery(
    "SELECT * FROM `category` WHERE status = 'active' AND is_featured = 1 LIMIT 12",
    next
  )
    .then((value) => {
      featuredCategory = value;
    })
    .catch((err) => {
      return next(new AppError(err, 500));
    });

  await promiseQuery(
    "SELECT * FROM `posts` WHERE status = 'active' LIMIT 9",
    next
  )
    .then((value) => {
      posts = value;
    })
    .catch((err) => {
      return next(new AppError(err, 500));
    });

  await promiseQuery(
    "SELECT * FROM `posts` WHERE status = 'active' AND is_trending = 1 LIMIT 18",
    next
  )
    .then((value) => {
      trendingPosts = value;
    })
    .catch((err) => {
      return next(new AppError(err, 500));
    });

  // console.log("catDAta ", catDAta);

  res.status(201).json({
    status: "success",
    message: "Home page configuration data",
    data: {
      categories,
      posts,
      featuredCategory,
      trendingPosts,
    },
  });
};

const promiseQuery = async (sqlQuery, next) => {
  const query = util.promisify(conn.query).bind(conn);

  try {
    return await query(sqlQuery);
  } catch (err) {
    return next(new AppError(err, 500));
  } finally {
  }
};
