const jwt = require('jsonwebtoken');
const getEnvConfig = require('../../utils/Config');

// Get JWT secret - use PRIVATE_KEY (standard) or fallback to JWT_SECRET
const JWT_SECRET = getEnvConfig('PRIVATE_KEY') || process.env.JWT_SECRET;

const authenticateToken = (req, res, next) => {
  try {
    if (!JWT_SECRET) {
      console.error('ERROR: JWT_SECRET or PRIVATE_KEY environment variable is required');
      return res.status(500).json({ error: 'Server configuration error' });
    }

    // Get token from Authorization header or from body/query
    let token = null;
    
    // Check Authorization header (Bearer token)
    const authHeader = req.headers['authorization'];
    if (authHeader && authHeader.startsWith('Bearer ')) {
      token = authHeader.substring(7);
    }
    
    // Fallback: check x-auth-token header
    if (!token && req.headers['x-auth-token']) {
      token = req.headers['x-auth-token'];
    }
    
    // Fallback: check body or query
    if (!token && req.body && req.body.token) {
      token = req.body.token;
    }
    
    if (!token && req.query && req.query.token) {
      token = req.query.token;
    }

    if (!token) {
      return res.status(401).json({ error: 'Access token is required' });
    }

    // Verify token
    const decoded = jwt.verify(token, JWT_SECRET);
    
    // Set user data in request
    req.user = decoded;
    
    next();
  } catch (error) {
    if (error.name === 'TokenExpiredError') {
      return res.status(401).json({ error: 'Token has expired' });
    }
    if (error.name === 'JsonWebTokenError') {
      return res.status(401).json({ error: 'Invalid token' });
    }
    console.error('Token authentication error:', error);
    return res.status(500).json({ error: 'Internal server error' });
  }
};

module.exports = authenticateToken;

