const express = require("express");
const attendance = require("../../controllers/frontend/attendance");
const branch = require("../../controllers/frontend/branch");
const roster = require("../../controllers/frontend/roster");
const staff = require("../../controllers/frontend/staff");
const availability = require("../../controllers/frontend/staffAvailability");
const staffAuth = require("../../controllers/frontend/staffAuth");
const featureFlags = require("../../controllers/frontend/featureFlags");
const { isUserRegistered, authenticateToken } = require("../../middleware/frontend");

const MobileRoutes = express.Router();

const authorizeRoles =
  (...allowedRoles) =>
  (req, res, next) => {
    if (!req.user || !allowedRoles.includes(req.user.role)) {
      return res.status(403).json({ error: "Access denied" });
    }
    return next();
  };

// Staff authentication routes (public)

MobileRoutes.route('/staff-login').post( isUserRegistered, staffAuth.staffLogin);
MobileRoutes.route('/refresh-token').post( isUserRegistered, staffAuth.refreshToken);
MobileRoutes.route('/register').post( isUserRegistered, staffAuth.register);
MobileRoutes.route('/forgot-password').post( isUserRegistered, staffAuth.forgotPassword);
MobileRoutes.route('/reset-password').get( isUserRegistered, staffAuth.resetPassword);
MobileRoutes.route('/reset-password').post( isUserRegistered, staffAuth.resetPassword);
MobileRoutes.route('/verify-email').get( isUserRegistered, staffAuth.verifyEmail);
MobileRoutes.route('/verify-email').post( isUserRegistered, staffAuth.verifyEmail);

// Protected staff routes
MobileRoutes.route("/profile").get( authenticateToken, staffAuth.getProfile);
MobileRoutes.route("/change-password").post(authenticateToken, staffAuth.changePassword);

// Attendance routes
MobileRoutes.route("/attendance/clock-in").post(
  isUserRegistered,
  attendance.clockIn,
);
MobileRoutes.route("/attendance/clock-out").post(
  isUserRegistered,
  attendance.clockOut,
);
MobileRoutes.route("/attendance/break-clock-in").post(
  isUserRegistered,
  attendance.breakClockIn,
);
MobileRoutes.route("/attendance/break-clock-out").post(
  isUserRegistered,
  attendance.breakClockOut,
);

MobileRoutes.route("/attendance/manual").post(
  isUserRegistered,
  attendance.markManualAttendance,
);
MobileRoutes.route("/attendance/manual-attendance").post(
  isUserRegistered,
  attendance.createOrUpdateManualAttendance,
);
MobileRoutes.route("/attendance/branch/:branch_id").get(
  isUserRegistered,
  attendance.getAttendance,
);
MobileRoutes.route("/attendance/branch/:branch_id/reports").get(
  isUserRegistered,
  attendance.getBranchReports,
);
MobileRoutes.route("/attendance/staff/:staff_id").get(
  isUserRegistered,
  attendance.getStaffAttendance,
);
MobileRoutes.route("/attendance/branch/:branch_id/summary").get(
  isUserRegistered,
  attendance.getAttendanceSummary,
); 
MobileRoutes.route("/attendance/breaks/:attendance_id").get(
  isUserRegistered,
  attendance.getBreaks,
);
MobileRoutes.route("/attendance/breaks").post(
  isUserRegistered,
  attendance.addBreak,
);
MobileRoutes.route("/attendance/breaks/:break_id").put(
  isUserRegistered,
  attendance.updateBreak,
);
MobileRoutes.route("/attendance/breaks/:break_id").delete(
  isUserRegistered,
  attendance.deleteBreak,
);


// Branch endpoints (optimized for mobile clients)
MobileRoutes.route("/branches").post(
  isUserRegistered,
  branch.createBranch
);

MobileRoutes.route("/branches/shop/:shop_id").get(
  isUserRegistered,
  branch.getBranchesByShop
);

MobileRoutes.route("/branch/:id").get(
  isUserRegistered,
  branch.getBranchById
);

MobileRoutes.route("/branch/:id").put(
  isUserRegistered,
  branch.updateBranch
);

MobileRoutes.route("/branch/:id").delete(
  isUserRegistered,
  branch.deleteBranch
);

MobileRoutes.route("/branches/:id/verify-pin").post(
  isUserRegistered,
  branch.verifyBranchPin
);


// Roster endpoints
MobileRoutes.route("/branch/:branch_id/config").get(
  isUserRegistered,
  roster.getConfig
);

MobileRoutes.route("/branch/:branch_id/shift-types").post(
  isUserRegistered,
  roster.createShiftType
);

MobileRoutes.route("/shift-types/:id").put(
  isUserRegistered,
  roster.updateShiftType
);

MobileRoutes.route("/shift-types/:id").delete(
  isUserRegistered,
  roster.deleteShiftType
);

MobileRoutes.route("/branch/:branch_id/positions").post(
  isUserRegistered,
  roster.createPosition
);

MobileRoutes.route("/positions/:id").delete(
  isUserRegistered,
  roster.deletePosition
);

MobileRoutes.route("/branch/:branch_id/break-types").post(
  isUserRegistered,
  roster.createBreakType
);

MobileRoutes.route("/break-types/:id").put(
  isUserRegistered,
  roster.updateBreakType
);

MobileRoutes.route("/break-types/:id").delete(
  isUserRegistered,
  roster.deleteBreakType
);

MobileRoutes.route("/branch/:branch_id/assignments").get(
  isUserRegistered,
  roster.getAssignments
);

MobileRoutes.route("/branch/:branch_id/assignments").post(
  isUserRegistered,
  roster.upsertAssignments
);

// Staff endpoints
MobileRoutes.route("/staff").post(
  isUserRegistered,
  staff.createStaff
);

MobileRoutes.route("/staff/branch/:branch_id").get(
  isUserRegistered,
  staff.getStaffByBranch
);

MobileRoutes.route("/staff/:id").get(
  isUserRegistered,
  staff.getStaffById
);

MobileRoutes.route("/staff/:id").put(
  isUserRegistered,
  staff.updateStaff
);

MobileRoutes.route("/staff/:id").delete(
  isUserRegistered,
  staff.deleteStaff
);

MobileRoutes.route("/staff/:id/face-embeddings").put(
  isUserRegistered,
  staff.updateFaceEmbeddings
);

// Staff availability endpoints
MobileRoutes.route("/availability/staff/:staff_id").get(
  isUserRegistered,
  availability.getStaffAvailability
);

MobileRoutes.route("/availability/staff/:staff_id").post(
  isUserRegistered,
  availability.upsertStaffAvailability
);

MobileRoutes.route("/availability/branch/:branch_id").get(
  isUserRegistered,
  availability.getBranchAvailability
);

// Feature flags
// Public route - check if a feature is enabled (no auth required)
MobileRoutes.route("/feature-flags/check/:featureName").get(
  isUserRegistered,
  featureFlags.checkFeatureFlag,
);

// Protected routes - only superadmin can access
MobileRoutes.route("/feature-flags").get(
  isUserRegistered,
  featureFlags.getFeatureFlags,
);
MobileRoutes.route("/feature-flags/:featureName").get(
  isUserRegistered,
  featureFlags.getFeatureFlag,
);
MobileRoutes.route("/feature-flags/:featureName").put(
  isUserRegistered,
  featureFlags.updateFeatureFlag,
);

module.exports = MobileRoutes;

