/**
 * Script to create Stripe Products and Prices for subscription plans
 * Run this script once to set up Stripe products and prices for existing subscription plans
 * 
 * Usage: node scripts/setupStripeProducts.js
 */

// Load environment variables from .env.local or .env
const fs = require('fs');
const path = require('path');

// Try to load .env.local first, then fall back to .env
const envLocalPath = path.join(__dirname, '..', '.env.local');
const envPath = path.join(__dirname, '..', '.env');

if (fs.existsSync(envLocalPath)) {
  console.log("Loading environment variables from .env.local");
  require('dotenv').config({ path: envLocalPath });
} else if (fs.existsSync(envPath)) {
  console.log("Loading environment variables from .env");
  require('dotenv').config({ path: envPath });
} else {
  console.log("Warning: Neither .env.local nor .env file found. Using system environment variables.");
  require('dotenv').config();
}

const stripe = require("stripe")(process.env.STRIPE_SECRET_KEY);
const mysql = require("mysql2");
const util = require("util");

// Create database connection for script
const DB_CONFIG = {
  host: process.env.DB_HOST,
  user: process.env.DB_USER,
  port: process.env.DB_PORT || 3306,
  password: process.env.DB_PASSWORD,
  database: process.env.DB_NAME,
};

// Validate required environment variables
console.log("Checking environment variables...");
console.log("DB_HOST:", DB_CONFIG.host ? "✓ Set" : "✗ Missing");
console.log("DB_USER:", DB_CONFIG.user ? "✓ Set" : "✗ Missing");
console.log("DB_NAME:", DB_CONFIG.database ? `✓ Set (${DB_CONFIG.database})` : "✗ Missing");
console.log("STRIPE_SECRET_KEY:", process.env.STRIPE_SECRET_KEY ? "✓ Set" : "✗ Missing");
console.log("");

if (!DB_CONFIG.host || !DB_CONFIG.user || !DB_CONFIG.database) {
  console.error("Error: Missing required database configuration in .env file");
  console.error("Required: DB_HOST, DB_USER, DB_NAME");
  console.error("\nPlease check your .env file in the large-app-api directory.");
  process.exit(1);
}

if (!process.env.STRIPE_SECRET_KEY) {
  console.error("Error: STRIPE_SECRET_KEY not found in .env file");
  process.exit(1);
}

const conn = mysql.createConnection(DB_CONFIG);

// Promisify query for use in script
const query = util.promisify(conn.query).bind(conn);

async function setupStripeProducts() {
  try {
    // Connect to database (only if not already connected)
    if (conn.state === 'disconnected') {
      await new Promise((resolve, reject) => {
        conn.connect((error) => {
          if (error) {
            console.error("Database connection error:", error.message);
            reject(error);
          } else {
            console.log(`Successfully connected to database: ${DB_CONFIG.database}`);
            resolve();
          }
        });
      });
    } else {
      console.log("Database connection already established.");
    }

    // Verify database connection by selecting the database explicitly
    await query(`USE ${mysql.escapeId(DB_CONFIG.database)}`);
    console.log(`Using database: ${DB_CONFIG.database}\n`);

    console.log("Starting Stripe Products and Prices setup...\n");

    // Fetch all active subscription plans from database (including Stripe IDs if they exist)
    const plansQuery = `
      SELECT id, name, price, duration_type, description, stripe_product_id, stripe_price_id
      FROM subscription_plans 
      WHERE status = 'active' AND is_deleted IS NULL
    `;

    const plans = await query(plansQuery);

    if (!plans || plans.length === 0) {
      console.log("No active subscription plans found in database.");
      return;
    }

    console.log(`Found ${plans.length} subscription plan(s) to process.\n`);

    for (const plan of plans) {
      try {
        console.log(`Processing plan: ${plan.name} (ID: ${plan.id})`);

        let productId = plan.stripe_product_id;
        let priceId = plan.stripe_price_id;

        // Create or get Stripe Product
        if (!productId) {
          const product = await stripe.products.create({
            name: plan.name,
            description: plan.description || `Subscription plan: ${plan.name}`,
            metadata: {
              plan_id: plan.id.toString(),
              duration_type: plan.duration_type,
            },
          });
          productId = product.id;
          console.log(`  ✓ Created Stripe Product: ${productId}`);
        } else {
          console.log(`  ✓ Using existing Stripe Product: ${productId}`);
        }

        // Create or get Stripe Price
        if (!priceId) {
          const amountInCents = Math.round(parseFloat(plan.price) * 100);
          
          // Determine interval based on duration_type
          const interval = plan.duration_type === "year" ? "year" : "month";
          const intervalCount = 1;

          const price = await stripe.prices.create({
            product: productId,
            unit_amount: amountInCents,
            currency: "usd",
            recurring: {
              interval: interval,
              interval_count: intervalCount,
            },
            metadata: {
              plan_id: plan.id.toString(),
              duration_type: plan.duration_type,
            },
          });
          priceId = price.id;
          console.log(`  ✓ Created Stripe Price: ${priceId} (${plan.price} USD/${interval})`);
        } else {
          console.log(`  ✓ Using existing Stripe Price: ${priceId}`);
        }

        // Update database with Stripe IDs
        const updateQuery = `
          UPDATE subscription_plans 
          SET stripe_product_id = ?, stripe_price_id = ? 
          WHERE id = ?
        `;

        await query(updateQuery, [productId, priceId, plan.id]);
        console.log(`  ✓ Updated database with Stripe IDs\n`);

      } catch (error) {
        console.error(`  ✗ Error processing plan ${plan.name}:`, error.message);
        continue;
      }
    }

    console.log("Stripe Products and Prices setup completed!");

  } catch (error) {
    console.error("Error in setupStripeProducts:", error);
    if (error.code) {
      console.error(`Error code: ${error.code}`);
      console.error(`Error message: ${error.message}`);
    }
    process.exit(1);
  } finally {
    // Close database connection
    if (conn && typeof conn.end === 'function') {
      conn.end((err) => {
        if (err) {
          console.error("Error closing database connection:", err);
        } else {
          console.log("Database connection closed.");
        }
        process.exit(0);
      });
    } else {
      process.exit(0);
    }
  }
}

// Run the setup
setupStripeProducts();

