// Load environment variables first (safety measure)
// This ensures env vars are loaded even if dotenv hasn't been initialized yet
require("dotenv").config({ path: process.env.DOTENV_CONFIG_PATH || ".env.local" });
if (!process.env.DB_HOST && !process.env.DB_USER) {
  require("dotenv").config({ path: ".env" });
}

const mysql = require("mysql2");

// Validate required environment variables
const requiredEnvVars = {
  DB_HOST: process.env.DB_HOST,
  DB_USER: process.env.DB_USER,
  DB_PASSWORD: process.env.DB_PASSWORD,
  DB_NAME: process.env.DB_NAME,
};

// Check for missing environment variables
// Note: DB_PASSWORD can be empty (empty string is valid for MySQL)
const missingVars = Object.entries(requiredEnvVars)
  .filter(([key, value]) => {
    if (key === 'DB_PASSWORD') {
      // Allow empty password (value === undefined or null means missing, but empty string is valid)
      return value === undefined || value === null;
    }
    // For other vars, check if they exist and are not empty
    return !value || (typeof value === 'string' && value.trim() === '');
  })
  .map(([key]) => key);

if (missingVars.length > 0) {
  console.error("❌ Missing required database environment variables:");
  missingVars.forEach(varName => {
    console.error(`   - ${varName}`);
  });
  console.error("\nPlease ensure your .env file contains all required database variables:");
  console.error("   DB_HOST, DB_USER, DB_PASSWORD, DB_NAME, DB_PORT (optional, defaults to 3306)");
  throw new Error(`Missing required database environment variables: ${missingVars.join(', ')}`);
}

const DB_CONFIG = {
  host: process.env.DB_HOST,
  user: process.env.DB_USER,
  port: process.env.DB_PORT || 3306,
  password: process.env.DB_PASSWORD,
  database: process.env.DB_NAME,
  waitForConnections: true,
  connectionLimit: 10,
  queueLimit: 0,
  // Keep connections alive to prevent timeout
  enableKeepAlive: true,
  keepAliveInitialDelay: 0,
  // Connection timeout settings (in milliseconds)
  connectTimeout: 60000, // 60 seconds to establish connection
  // Prevent connections from being closed due to inactivity
  // MySQL default wait_timeout is 28800 seconds (8 hours)
  // We'll ping connections every 4 hours to keep them alive
  acquireTimeout: 60000, // 60 seconds to get connection from pool
  timeout: 60000, // 60 seconds query timeout
};

// Use connection pool instead of single connection
// This handles connection drops and automatically reconnects
const pool = mysql.createPool(DB_CONFIG);

// Test the connection
pool.getConnection((error, connection) => {
  if (error) {
    console.error("❌ Database connection error:", error.message);
    console.error(`   Attempted connection to: ${DB_CONFIG.host}:${DB_CONFIG.port}`);
    console.error(`   Database: ${DB_CONFIG.database}`);
    console.error(`   User: ${DB_CONFIG.user || '(empty)'}`);
    console.error("\nPlease check:");
    console.error("   1. Database server is running");
    console.error("   2. Environment variables are set correctly in .env file");
    console.error("   3. Database user has proper permissions");
    console.error("   4. Database name exists");
    throw error;
  }
  console.log(`✅ Successfully connected to the database: ${DB_CONFIG.database}@${DB_CONFIG.host}`);
  connection.release(); // Release the connection back to the pool
});

// Handle pool errors
pool.on('error', (err) => {
  console.error('❌ Database pool error:', err);
  if (err.code === 'PROTOCOL_CONNECTION_LOST') {
    console.error('Database connection was closed. Pool will automatically reconnect.');
  } else if (err.fatal) {
    console.error('Fatal database error:', err);
  }
});

// Keep connections alive by pinging them periodically
// This prevents connections from timing out after MySQL's wait_timeout (default 8 hours)
// We ping every 4 hours to ensure connections stay alive
const KEEP_ALIVE_INTERVAL = 4 * 60 * 60 * 1000; // 4 hours in milliseconds
let connectionStartTime = Date.now();
let lastPingTime = null;
let pingCount = 0;

const performKeepAlivePing = () => {
  pool.getConnection((err, connection) => {
    if (err) {
      console.error('❌ Error getting connection for keep-alive ping:', err.message);
      return;
    }
    
    // Ping the connection to keep it alive
    connection.ping((pingErr) => {
      connection.release(); // Always release the connection back to pool
      lastPingTime = new Date().toISOString();
      pingCount++;
      
      if (pingErr) {
        console.error('❌ Error pinging database connection:', pingErr.message);
      } else {
        const uptime = Math.floor((Date.now() - connectionStartTime) / (60 * 60 * 1000));
        console.log(`✅ Database keep-alive ping #${pingCount} successful (Uptime: ${uptime} hours, Last ping: ${lastPingTime})`);
      }
    });
  });
};

// Start keep-alive immediately and then every 4 hours
performKeepAlivePing(); // First ping immediately
const keepAliveInterval = setInterval(performKeepAlivePing, KEEP_ALIVE_INTERVAL);

// Log when keep-alive starts
console.log(`🔄 Database keep-alive ping started. Will ping every ${KEEP_ALIVE_INTERVAL / (60 * 60 * 1000)} hours`);
console.log(`📊 Connection pool stats: limit=${DB_CONFIG.connectionLimit}, queueLimit=${DB_CONFIG.queueLimit}`);

// Export connection health check function
const getConnectionHealth = () => {
  return {
    status: 'healthy',
    uptime: Math.floor((Date.now() - connectionStartTime) / 1000), // seconds
    uptimeHours: Math.floor((Date.now() - connectionStartTime) / (60 * 60 * 1000)),
    lastPingTime: lastPingTime,
    pingCount: pingCount,
    poolConfig: {
      connectionLimit: DB_CONFIG.connectionLimit,
      queueLimit: DB_CONFIG.queueLimit,
    }
  };
};

// Clean up interval on process exit
process.on('SIGINT', () => {
  clearInterval(keepAliveInterval);
  pool.end(() => {
    console.log('Database pool closed.');
    process.exit(0);
  });
});

process.on('SIGTERM', () => {
  clearInterval(keepAliveInterval);
  pool.end(() => {
    console.log('Database pool closed.');
    process.exit(0);
  });
});

// Store active transaction connection per async context
// Note: This is a simplified approach for backward compatibility
// For production, consider refactoring to use pool.getConnection() explicitly
let activeTransactionConnection = null;

// Export pool instead of single connection
// For backward compatibility, we'll create a wrapper that uses the pool
// This wrapper provides the same interface as a single connection
const connection = {
  query: function(sql, params, callback) {
    // If we're in a transaction, use the transaction connection
    if (activeTransactionConnection) {
      return activeTransactionConnection.query(sql, params, callback);
    }
    // Otherwise use the pool
    return pool.query(sql, params, callback);
  },
  beginTransaction: function(callback) {
    // Get a connection from the pool for the transaction
    pool.getConnection((err, conn) => {
      if (err) {
        if (callback) return callback(err);
        throw err;
      }
      conn.beginTransaction((err) => {
        if (err) {
          conn.release();
          if (callback) return callback(err);
          throw err;
        }
        // Store the connection for this transaction
        activeTransactionConnection = conn;
        if (callback) callback(null);
      });
    });
  },
  commit: function(callback) {
    if (!activeTransactionConnection) {
      const err = new Error('No active transaction to commit');
      if (callback) return callback(err);
      throw err;
    }
    const conn = activeTransactionConnection;
    activeTransactionConnection = null; // Clear before commit
    conn.commit((err) => {
      conn.release(); // Always release the connection
      if (callback) callback(err);
    });
  },
  rollback: function(callback) {
    if (!activeTransactionConnection) {
      const err = new Error('No active transaction to rollback');
      if (callback) return callback(err);
      throw err;
    }
    const conn = activeTransactionConnection;
    activeTransactionConnection = null; // Clear before rollback
    conn.rollback((err) => {
      conn.release(); // Always release the connection
      if (callback) callback(err);
    });
  },
  end: function(callback) {
    return pool.end(callback);
  },
  getConnection: function(callback) {
    return pool.getConnection(callback);
  }
};

// Export connection and health check
module.exports = connection;
module.exports.getConnectionHealth = getConnectionHealth;
module.exports.pool = pool; // Export pool for advanced usage if needed
