const crypto = require('crypto');
require('dotenv').config();

// Encryption configuration
const ALGORITHM = 'aes-256-gcm';
const IV_LENGTH = 16; // For AES, this is always 16
const SALT_LENGTH = 64;
const TAG_LENGTH = 16;
const KEY_LENGTH = 32; // 256 bits

/**
 * Derives a cryptographic key from the encryption key in environment
 * @returns {Buffer} Derived key
 */
function getEncryptionKey() {
  const encryptionKey = process.env.ENCRYPTION_KEY;
  
  if (!encryptionKey) {
    throw new Error('ENCRYPTION_KEY not found in environment variables');
  }
  
  if (encryptionKey.length < 32) {
    throw new Error('ENCRYPTION_KEY must be at least 32 characters long');
  }
  
  // Use a fixed salt for key derivation (stored in env or hardcoded)
  // In production, consider using a separate ENCRYPTION_SALT env variable
  const salt = crypto.createHash('sha256').update('faceml-staff-encryption-salt').digest();
  
  // Derive key using scrypt
  const key = crypto.scryptSync(encryptionKey, salt, KEY_LENGTH);
  return key;
}

/**
 * Encrypts a string or object using AES-256-GCM
 * @param {string|object} data - Data to encrypt
 * @returns {string} Encrypted data in format: iv:encrypted:authTag
 */
function encrypt(data) {
  try {
    if (data === null || data === undefined) {
      return null;
    }
    
    // Convert object to JSON string if needed
    const text = typeof data === 'object' ? JSON.stringify(data) : String(data);
    
    // Generate random IV for each encryption
    const iv = crypto.randomBytes(IV_LENGTH);
    
    // Get encryption key
    const key = getEncryptionKey();
    
    // Create cipher
    const cipher = crypto.createCipheriv(ALGORITHM, key, iv);
    
    // Encrypt the data
    let encrypted = cipher.update(text, 'utf8', 'hex');
    encrypted += cipher.final('hex');
    
    // Get authentication tag
    const authTag = cipher.getAuthTag();
    
    // Return format: iv:encrypted:authTag (all in hex)
    return `${iv.toString('hex')}:${encrypted}:${authTag.toString('hex')}`;
  } catch (error) {
    console.error('Encryption error:', error.message);
    throw new Error('Failed to encrypt data');
  }
}

/**
 * Decrypts data encrypted with encrypt function
 * @param {string} encryptedData - Encrypted data in format: iv:encrypted:authTag
 * @returns {string} Decrypted data
 */
function decrypt(encryptedData) {
  try {
    if (!encryptedData || encryptedData === null || encryptedData === 'null') {
      return null;
    }
    
    // Split the encrypted data
    const parts = encryptedData.split(':');
    if (parts.length !== 3) {
      throw new Error('Invalid encrypted data format');
    }
    
    const iv = Buffer.from(parts[0], 'hex');
    const encrypted = parts[1];
    const authTag = Buffer.from(parts[2], 'hex');
    
    // Get encryption key
    const key = getEncryptionKey();
    
    // Create decipher
    const decipher = crypto.createDecipheriv(ALGORITHM, key, iv);
    decipher.setAuthTag(authTag);
    
    // Decrypt the data
    let decrypted = decipher.update(encrypted, 'hex', 'utf8');
    decrypted += decipher.final('utf8');
    
    return decrypted;
  } catch (error) {
    console.error('Decryption error:', error.message);
    throw new Error('Failed to decrypt data');
  }
}

/**
 * Encrypts specific fields in an object
 * @param {object} obj - Object containing fields to encrypt
 * @param {string[]} fields - Array of field names to encrypt
 * @returns {object} Object with encrypted fields
 */
function encryptFields(obj, fields) {
  const result = { ...obj };
  
  fields.forEach(field => {
    if (result[field] !== undefined && result[field] !== null) {
      result[field] = encrypt(result[field]);
    }
  });
  
  return result;
}

/**
 * Decrypts specific fields in an object
 * @param {object} obj - Object containing encrypted fields
 * @param {string[]} fields - Array of field names to decrypt
 * @returns {object} Object with decrypted fields
 */
function decryptFields(obj, fields) {
  const result = { ...obj };
  
  fields.forEach(field => {
    if (result[field] !== undefined && result[field] !== null) {
      try {
        result[field] = decrypt(result[field]);
      } catch (error) {
        console.warn(`Failed to decrypt field ${field}:`, error.message);
        result[field] = null;
      }
    }
  });
  
  return result;
}

/**
 * Encrypts face embeddings (special handling for JSON)
 * @param {object|string} embeddings - Face embeddings object or JSON string
 * @returns {string} Encrypted embeddings
 */
function encryptFaceEmbeddings(embeddings) {
  if (!embeddings) return null;
  
  // If it's already a string, assume it's JSON
  const embeddingsStr = typeof embeddings === 'string' 
    ? embeddings 
    : JSON.stringify(embeddings);
  
  return encrypt(embeddingsStr);
}

/**
 * Decrypts face embeddings and returns as object
 * @param {string} encryptedEmbeddings - Encrypted embeddings
 * @returns {object|null} Decrypted embeddings object
 */
function decryptFaceEmbeddings(encryptedEmbeddings) {
  if (!encryptedEmbeddings) return null;
  
  try {
    const decrypted = decrypt(encryptedEmbeddings);
    if (!decrypted) return null;
    
    // Try to parse as JSON
    try {
      return JSON.parse(decrypted);
    } catch {
      // If not JSON, return as is
      return decrypted;
    }
  } catch (error) {
    console.warn('Failed to decrypt face embeddings:', error.message);
    return null;
  }
}

module.exports = {
  encrypt,
  decrypt,
  encryptFields,
  decryptFields,
  encryptFaceEmbeddings,
  decryptFaceEmbeddings
};
